#include "ApplePlatform.h"
#include "ASLog.hpp"
#include "AVPlayerSink.h"
#include "AudioConverterDecoder.hpp"
#include "AudioQueueRenderer.hpp"
#include "CocoaHttpClient.h"
#include "DisplayLayerRenderer.h"
#include "VTDecoder.hpp"
#include "media/ProtectionSystem.hpp"
#include "player/PassthroughSource.hpp"

namespace twitch {
ApplePlatform::ApplePlatform()
    : m_tempPath([NSTemporaryDirectory() UTF8String])
    , m_client(std::make_shared<CocoaHttpClient>())
    , m_readerFactory(std::make_shared<media::FileReaderFactory>())
    , m_log(std::make_shared<ASLog>())
    , m_passthroughMode(false)
{
    m_capabilities.avcFormat = AVCFormatType::AVCC;
    m_capabilities.supportsLowLatencyABR = true;

    UIUserInterfaceIdiom uiIdiom = [[UIDevice currentDevice] userInterfaceIdiom];
    switch (uiIdiom) {
    case UIUserInterfaceIdiomUnspecified:
    case UIUserInterfaceIdiomPhone:
    case UIUserInterfaceIdiomPad:
    default:
        m_name = "ios";
        break;
    case UIUserInterfaceIdiomTV:
        m_name = "tvos";
        break;
    }
}

std::unique_ptr<MediaDecoder> ApplePlatform::createDecoder(std::shared_ptr<const MediaFormat> format)
{
    if (format->getType().matches(MediaType::Audio_AAC)) {
        return std::unique_ptr<MediaDecoder>(new AudioConverterDecoder(m_log));
    } else if (format->getType().matches(MediaType::Video_AVC)) {
        return std::unique_ptr<MediaDecoder>(new VTDecoder(m_log));
    }

    return nullptr;
}

std::unique_ptr<MediaRenderer> ApplePlatform::createRenderer(const ReferenceClock& clock, std::shared_ptr<const MediaFormat> format)
{
    
    if (format->getType().matches(MediaType::Audio_AAC)) {
        return std::unique_ptr<MediaRenderer>(new AudioQueueRenderer(m_log));
    } else if (format->getType().matches(MediaType::Video_AVC)) {
        return std::unique_ptr<MediaRenderer>(new DisplayLayerRenderer(clock));
    }

    return nullptr;
}

std::unique_ptr<MediaSource> ApplePlatform::createSource(const std::string& path, const MediaType& mediaType,
    MediaSource::Listener& listener, std::shared_ptr<Scheduler> scheduler)
{
    if (m_passthroughMode) {
        (void)scheduler;
        if (MediaType::Video_MP4.matches(mediaType)) {
            return std::unique_ptr<MediaSource>(new PassthroughSource(listener, mediaType, path));
        }
    }
    return std::unique_ptr<MediaSource>(NativePlatform::createSource(path, mediaType, listener, scheduler));
}

std::unique_ptr<MediaSink> ApplePlatform::createSink(MediaSink::Listener& listener, std::shared_ptr<Scheduler> scheduler)
{
    if (m_passthroughMode) {
        return std::unique_ptr<AVPlayerSink>(new AVPlayerSink(listener, scheduler));
    } else {
        return std::unique_ptr<MediaSink>(NativePlatform::createSink(listener, scheduler));
    }
}

const std::set<MediaType>& ApplePlatform::getSupportedMediaTypes() const
{
    static const std::set<MediaType> passthroughTypes = { MediaType::Video_MP2T };
    if (m_passthroughMode) {
        return passthroughTypes;
    }
    return NativePlatform::getSupportedMediaTypes();
}

const std::set<std::vector<uint8_t>>& ApplePlatform::getSupportedProtectionSystems() const
{
    const static std::set<std::vector<uint8_t>> systemId = { media::ProtectionSystemFairPlay.toBytes() };
    return systemId;
}

void ApplePlatform::setCurrentThreadName(const std::string& name)
{
    [[NSThread currentThread] setName:[NSString stringWithUTF8String:name.c_str()]];
}

void ApplePlatform::setPassthroughMode(bool enable)
{
    m_passthroughMode = enable;
    m_readerFactory->setPassthrough(m_passthroughMode);
}

std::shared_ptr<IOSPlatform> IOSPlatform::create()
{
    return std::make_shared<ApplePlatform>();
}
    
UInt32 ApplePlatform::getAudioSessionSampleRate()
{
    AVAudioSession* audioSession = [AVAudioSession sharedInstance];
    return static_cast<UInt32>(audioSession.sampleRate);
}
}
