//
//  AudioConverterDecoder.cpp
//  player-core
//
//  Created by Purushe, Nikhil on 11/1/16.
//
//

#include "AudioConverterDecoder.hpp"

namespace twitch {
const int AudioConverterBufferEmptied = 'empt';

AudioConverterDecoder::AudioConverterDecoder(std::shared_ptr<Log> log)
    : m_log(log)
    , m_audioConverter(nullptr)
    , m_inputBuffers {}
    , m_outputBuffers {}
    , m_inputDescription {}
    , m_outputDescription {}
    , m_outputPacketDescription {}
    , m_outputFrameSize(0)
    , m_sampleTime(MediaTime::zero())
{
}

AudioConverterDecoder::~AudioConverterDecoder()
{
    if (m_audioConverter) {
        AudioConverterDispose(m_audioConverter);
        m_audioConverter = nullptr;
    }

    freeAudioBufferList(m_inputBuffers);
    freeAudioBufferList(m_outputBuffers);
}

MediaResult AudioConverterDecoder::configure(const MediaFormat& input, MediaFormat& output)
{
    if (!input.getType().matches(MediaType::Audio_AAC)) {
        return MediaResult::ErrorNotSupported;
    }

    UInt32 sampleRate = input.getInt(MediaFormat::Audio_SampleRate);
    UInt32 channelCount = input.getInt(MediaFormat::Audio_ChannelCount);
    UInt32 outputSampleRate = ApplePlatform::getAudioSessionSampleRate();

    if (m_audioConverter) {
        // reuse existing converter if possible
        if (m_inputDescription.mSampleRate == sampleRate
            && m_inputDescription.mChannelsPerFrame == channelCount
            && static_cast<UInt32>(m_outputDescription.mSampleRate) == outputSampleRate) {

            m_log->info("AudioQueueDecoder reset");
            setOutputFormat(output);
            OSStatus status = AudioConverterReset(m_audioConverter);
            return check(status, "AudioConverterReset");
        } else {
            AudioConverterDispose(m_audioConverter);
            m_audioConverter = nullptr;
        }
    }

    m_log->info("AudioQueueDecoder AAC sample rate %d channels %d", sampleRate, channelCount);

    m_inputDescription.mSampleRate = sampleRate;
    m_inputDescription.mFormatID = kAudioFormatMPEG4AAC;
    m_inputDescription.mFormatFlags = kMPEG4Object_AAC_LC;
    m_inputDescription.mBytesPerPacket = 0;
    m_inputDescription.mFramesPerPacket = 1024;
    m_inputDescription.mBytesPerFrame = 0;
    m_inputDescription.mChannelsPerFrame = channelCount;
    m_inputDescription.mBitsPerChannel = 0; // indicates compressed format
    m_inputDescription.mReserved = 0;

    m_outputDescription.mSampleRate = outputSampleRate;
    m_outputDescription.mFormatID = kAudioFormatLinearPCM;
    m_outputDescription.mFormatFlags = kLinearPCMFormatFlagIsSignedInteger;
    m_outputDescription.mFramesPerPacket = 1;
    m_outputDescription.mBytesPerFrame = 2 * channelCount;
    m_outputDescription.mBytesPerPacket = m_outputDescription.mBytesPerFrame;
    m_outputDescription.mChannelsPerFrame = channelCount;
    m_outputDescription.mBitsPerChannel = 16;
    m_outputDescription.mReserved = 0;

    freeAudioBufferList(m_inputBuffers);
    freeAudioBufferList(m_outputBuffers);
    createAudioBufferList(m_inputBuffers);
    createAudioBufferList(m_outputBuffers);

    OSStatus status = AudioConverterNew(&m_inputDescription, &m_outputDescription, &m_audioConverter);

    if (status != noErr) {
        return check(status, "AudioConverterNew");
    }

    // need full mp4 esds box for the decoder just generate it from another converter instance with the formats reversed
    AudioConverterRef tempConverter;
    status = AudioConverterNew(&m_outputDescription, &m_inputDescription, &tempConverter);

    if (status != noErr) {
        return check(status, "AudioConverterNew");
    }

    UInt32 cookieSize = 256;
    UInt8 cookieData[cookieSize];
    status = AudioConverterGetProperty(tempConverter, kAudioConverterCompressionMagicCookie, &cookieSize, cookieData);

    if (status != noErr) {
        AudioConverterDispose(tempConverter);
        return check(status, "AudioConverterGetProperty");
    }

    AudioConverterDispose(tempConverter);
    // set the cookie
    status = AudioConverterSetProperty(m_audioConverter, kAudioConverterDecompressionMagicCookie, cookieSize, cookieData);

    if (status != noErr) {
        return check(status, "Failed to set kAudioConverterDecompressionMagicCookie");
    }

    setOutputFormat(output);
    return MediaResult::Ok;
}

MediaResult AudioConverterDecoder::decode(const MediaSampleBuffer& input)
{
    if (!m_audioConverter) {
        return MediaResult::ErrorInvalidState;
    }

    m_sampleTime = input.decodeTime;
    UInt32 bufferSize = static_cast<UInt32>(input.buffer.size());
    memcpy(m_inputBuffers.mBuffers[0].mData, input.buffer.data(), bufferSize);
    m_inputBuffers.mBuffers[0].mDataByteSize = bufferSize;
    m_inputBuffers.mBuffers[0].mNumberChannels = m_inputDescription.mChannelsPerFrame;

    m_outputFrameSize = m_outputDescription.mSampleRate / m_outputDescription.mBytesPerFrame;

    OSStatus status = AudioConverterFillComplexBuffer(
        m_audioConverter, &inputCallback, this, &m_outputFrameSize, &m_outputBuffers, NULL);

    if (status != noErr && status != AudioConverterBufferEmptied) {
        return check(status, "AudioConverterFillComplexBuffer");
    }

    return MediaResult::Ok;
}

MediaResult AudioConverterDecoder::hasOutput(bool& hasOutput)
{
    hasOutput = m_outputFrameSize > 0;
    return MediaResult::Ok;
}

MediaResult AudioConverterDecoder::getOutput(std::shared_ptr<MediaSample>& output)
{
    uint8_t* data = static_cast<uint8_t*>(m_outputBuffers.mBuffers[0].mData);
    auto buffer = std::make_shared<MediaSampleBuffer>();
    buffer->buffer.assign(data, data + (m_outputFrameSize * m_outputDescription.mBytesPerFrame));
    buffer->decodeTime = m_sampleTime;
    buffer->presentationTime = m_sampleTime;
    output = buffer;
    m_outputFrameSize = 0; // signals no more output samples
    return MediaResult::Ok;
}

OSStatus AudioConverterDecoder::inputCallback(AudioConverterRef inAudioConverter,
    UInt32* ioNumberDataPackets,
    AudioBufferList* ioData,
    AudioStreamPacketDescription** outDataPacketDescription,
    void* inUserData)
{
    (void)inAudioConverter;
    AudioConverterDecoder* decoder = reinterpret_cast<AudioConverterDecoder*>(inUserData);
    UInt32 dataSize = decoder->m_inputBuffers.mBuffers[0].mDataByteSize;

    if (dataSize <= 0) {
        *ioNumberDataPackets = 0;
        return AudioConverterBufferEmptied;
    }

    ioData->mBuffers[0].mDataByteSize = dataSize;
    ioData->mBuffers[0].mData = decoder->m_inputBuffers.mBuffers[0].mData;
    ioData->mBuffers[0].mNumberChannels = decoder->m_inputBuffers.mBuffers[0].mNumberChannels;

    decoder->m_outputPacketDescription.mDataByteSize = ioData->mBuffers[0].mDataByteSize;
    decoder->m_outputPacketDescription.mStartOffset = 0;
    decoder->m_outputPacketDescription.mVariableFramesInPacket = 0;

    if (outDataPacketDescription) {
        *outDataPacketDescription = &decoder->m_outputPacketDescription;
    }

    decoder->m_inputBuffers.mBuffers[0].mDataByteSize = 0; // we only input 1 packet at a time
    return noErr;
}

MediaResult AudioConverterDecoder::flush()
{
    OSStatus status = AudioConverterReset(m_audioConverter);
    return check(status, "AudioConverterReset");
}

MediaResult AudioConverterDecoder::reset()
{
    OSStatus status = AudioConverterReset(m_audioConverter);
    return check(status, "AudioConverterReset");
}

void AudioConverterDecoder::createAudioBufferList(AudioBufferList& list)
{
    UInt32 sampleRate = static_cast<UInt32>(m_outputDescription.mSampleRate);
    list.mNumberBuffers = 1;
    list.mBuffers[0].mNumberChannels = 1;
    list.mBuffers[0].mData = new UInt8[sampleRate];
    list.mBuffers[0].mDataByteSize = sampleRate;
}

void AudioConverterDecoder::freeAudioBufferList(AudioBufferList& list)
{
    for (UInt32 i = 0; i < list.mNumberBuffers; i++) {
        UInt8* data = static_cast<UInt8*>(list.mBuffers[i].mData);
        if (data) {
            delete[] data;
        }
    }
}

void AudioConverterDecoder::setOutputFormat(MediaFormat& output)
{
    output.setType(MediaType::Audio_PCM);
    output.setInt(MediaFormat::Audio_SampleRate, static_cast<int>(m_outputDescription.mSampleRate));
    output.setInt(MediaFormat::Audio_ChannelCount, m_outputDescription.mChannelsPerFrame);
    output.setInt(MediaFormat::Audio_BitSize, m_outputDescription.mBitsPerChannel);
}

MediaResult AudioConverterDecoder::check(OSStatus status, const char* message)
{
    if (status != noErr) {
        m_log->error("error %d message %s", status, message);
        return MediaResult(MediaResult::Error, status);
    }

    return MediaResult::Ok;
}
}
