//
//  AudioQueueRenderer.cpp
//  player-core
//
//  Created by Purushe, Nikhil on 11/3/16.
//
//

#include "AudioQueueRenderer.hpp"
#include <pthread.h>

namespace twitch {
AudioQueueRenderer::AudioQueueRenderer(std::shared_ptr<Log> log)
    : m_log(log)
    , m_description {}
    , m_audioQueue(nullptr)
    , m_buffers {}
    , m_bufferIndex(0)
    , m_started(false)
    , m_primed(false)
    , m_initialSampleTime(MediaTime::zero())
    , m_playbackRate(1.0)
    , m_volume(1.0)
{
    // set thread priority to high
    int policy;
    struct sched_param param;
    pthread_getschedparam(pthread_self(), &policy, &param);
    param.sched_priority = sched_get_priority_max(policy);
    pthread_setschedparam(pthread_self(), policy, &param);
}

AudioQueueRenderer::~AudioQueueRenderer()
{
    m_started = false;
    m_bufferAvailable.notify_one();
    disposeQueue();
}

MediaResult AudioQueueRenderer::configure(const MediaFormat& format)
{
    m_log->info("AudioQueueRenderer::configure");

    UInt32 sampleRate = format.getInt(MediaFormat::Audio_SampleRate);
    UInt32 channelCount = format.getInt(MediaFormat::Audio_ChannelCount);

    // if the queue is already created only recreate if the samplerate or channel count is changing
    if (m_audioQueue
        && sampleRate == m_description.mSampleRate
        && channelCount == m_description.mChannelsPerFrame) {
        return MediaResult::Ok;
    }

    m_description.mSampleRate = sampleRate;
    m_description.mFormatID = kAudioFormatLinearPCM;
    m_description.mFormatFlags = kLinearPCMFormatFlagIsSignedInteger;
    m_description.mFramesPerPacket = 1;
    m_description.mBytesPerFrame = 2 * channelCount;
    m_description.mBytesPerPacket = m_description.mBytesPerFrame;
    m_description.mChannelsPerFrame = channelCount;
    m_description.mBitsPerChannel = 16;
    m_description.mReserved = 0;

    // reset audio queue
    disposeQueue();

    m_log->info("AudioQueue create new output");
    OSStatus status = AudioQueueNewOutput(&m_description, &outputCallback, this, NULL, NULL, 0, &m_audioQueue);

    if (status != noErr) {
        return check(status, "AudioQueueNewOutput");
    }

    setPlaybackRate(m_playbackRate);
    setVolume(m_volume);

    for (int i = 0; i < AudioBufferCount; i++) {
        status = AudioQueueAllocateBuffer(m_audioQueue, sampleRate * m_description.mBytesPerFrame, &m_buffers[i]);

        if (status != noErr) {
            return check(status, "AudioQueueAllocateBuffer");
        }
    }

    status = AudioQueueAddPropertyListener(m_audioQueue, kAudioQueueProperty_IsRunning, &propertyListenerCallback, this);
    check(status, "AudioQueueAddPropertyListener");

    // check device output sample rate matches the queue setting
    Float64 deviceSampleRate = 0.0;
    UInt32 size = sizeof(deviceSampleRate);
    status = AudioQueueGetProperty(m_audioQueue, kAudioQueueDeviceProperty_SampleRate, &deviceSampleRate, &size);
    check(status, "AudioQueueGetProperty kAudioQueueDeviceProperty_SampleRate");
    if (status == noErr && deviceSampleRate != m_description.mSampleRate) {
        m_log->warn("AudioQueue sample rate %f != device sample rate %f", m_description.mSampleRate, deviceSampleRate);
    }

    return MediaResult::Ok;
}

MediaResult AudioQueueRenderer::render(std::shared_ptr<const MediaSample> input)
{
    if (!m_audioQueue) {
        return MediaResult::ErrorInvalidState;
    }

    if (m_initialSampleTime == MediaTime::zero() || m_initialSampleTime > input->presentationTime) {
        m_initialSampleTime = input->presentationTime;
    }

    std::unique_lock<std::mutex> lock(m_mutex);

    // block until the input queue is drained
    if (m_started && m_inputQueue.size() >= AudioBufferCount) {
        const auto timeout = std::chrono::seconds(1);
        if (!m_bufferAvailable.wait_for(lock, timeout, [this]() { return m_inputQueue.size() < AudioBufferCount || !m_started; })) {
            m_log->warn("AudioQueueRenderer buffer available wait timed out");
            OSStatus status = AudioQueueStop(m_audioQueue, true);
            check(status, "AudioQueueStop");
            resetState();
        }
    }

    m_inputQueue.push(input);

    if (!m_primed) {
        if (m_bufferIndex >= AudioBufferCount) {
            m_bufferIndex = 0;
        }
        fillBuffer(m_buffers[m_bufferIndex++]);

        if (m_bufferIndex == AudioBufferCount) {
            m_primed = true;
        }
    }

    if (!m_started && m_primed && m_inputQueue.size() >= AudioBufferCount) {
        OSStatus status = AudioQueueStart(m_audioQueue, NULL);

        if (status != noErr) {
            return check(status, "AudioQueueStart");
        }

        m_started = true;
    }

    return MediaResult::Ok;
}

MediaResult AudioQueueRenderer::start()
{
    if (!m_audioQueue) {
        m_log->info("AudioQueueDecoder::start() with no AudioQueue");
        return MediaResult::ErrorInvalidState;
    }

    // queue cannot be actually started until all buffers are enqueued.
    return MediaResult::Ok;
}

MediaResult AudioQueueRenderer::stop()
{
    if (!m_audioQueue) {
        return MediaResult::ErrorInvalidState;
    }

    m_log->info("AudioQueueRenderer::stop");
    MediaResult result = MediaResult::Ok;

    if (m_started) {
        OSStatus status = AudioQueueStop(m_audioQueue, true);
        result = check(status, "AudioQueueStop");
        resetState();
    }

    return result;
}

void AudioQueueRenderer::disposeQueue()
{
    if (m_audioQueue) {
        AudioQueueRemovePropertyListener(m_audioQueue, kAudioQueueProperty_IsRunning, &propertyListenerCallback, this);

        {
            std::lock_guard<std::mutex> lock(m_mutex);
            m_inputQueue = std::queue<std::shared_ptr<const MediaSample>>();
        }

        check(AudioQueueStop(m_audioQueue, true), "AudioQueueStop");
        check(AudioQueueDispose(m_audioQueue, true), "AudioQueueDispose");

        for (int i = 0; i < AudioBufferCount; i++) {
            if (m_buffers[i]) {
                m_buffers[i] = nullptr;
            }
        }

        m_audioQueue = nullptr;
    }
    resetState();
}

void AudioQueueRenderer::resetState()
{
    m_started = false;
    m_primed = false;
    m_bufferIndex = 0;
    m_initialSampleTime = MediaTime::zero();
    m_lastPresentationTime = MediaTime::invalid();
}

void AudioQueueRenderer::fillBuffer(AudioQueueBufferRef inBuffer)
{
    if (m_inputQueue.empty()) {
        // if we're unable to fill an output buffer the output callback won't get called
        if (m_started) {
            OSStatus status = AudioQueueStop(m_audioQueue, true);
            check(status, "AudioQueueStop");
            resetState();
        }
        return;
    }

    inBuffer->mAudioDataByteSize = 0;
    inBuffer->mUserData = nullptr;

    auto sample = std::static_pointer_cast<const MediaSampleBuffer>(m_inputQueue.front());
    m_lastPresentationTime = sample->presentationTime;
    const UInt8* buffer = sample->buffer.data();
    UInt32 bufferSize = static_cast<UInt32>(sample->buffer.size());

    if (inBuffer->mAudioDataBytesCapacity < bufferSize) {
        m_log->error("Audio buffer too small have %d needed %d", inBuffer->mAudioDataBytesCapacity, bufferSize);
        return;
    }

    UInt8* inAudioData = static_cast<UInt8*>(inBuffer->mAudioData);
    std::memcpy(inAudioData + inBuffer->mAudioDataByteSize, buffer, bufferSize);

    inBuffer->mAudioDataByteSize += bufferSize;

    m_inputQueue.pop();

    OSStatus status = AudioQueueEnqueueBuffer(m_audioQueue, inBuffer, 0, NULL);

    if (status == kAudioQueueErr_EnqueueDuringReset) {
        m_log->warn("reset during enqueue (kAudioQueueErr_EnqueueDuringReset)");
        return;
    }
    check(status, "AudioQueueEnqueueBuffer");
}

MediaResult AudioQueueRenderer::flush()
{
    m_log->info("AudioQueueRenderer::flush()");
    if (!m_audioQueue) {
        return MediaResult::ErrorInvalidState;
    }

    {
        std::lock_guard<std::mutex> lock(m_mutex);
        m_inputQueue = std::queue<std::shared_ptr<const MediaSample>>();
        resetState();
    }

    OSStatus status = AudioQueueStop(m_audioQueue, true);
    return check(status, "AudioQueueStop");
}

MediaResult AudioQueueRenderer::setVolume(float volume)
{
    if (!m_audioQueue) {
        m_volume = volume;
        return MediaResult::Ok;
    } else {
        OSStatus status = AudioQueueSetParameter(m_audioQueue, kAudioQueueParam_Volume, volume);
        return check(status, "AudioQueueSetParameter kAudioQueueParam_Volume");
    }
}

MediaResult AudioQueueRenderer::setPlaybackRate(float rate)
{
    // https://developer.apple.com/reference/audiotoolbox/kaudioqueueparam_playrate
    static const float MinPlaybackRate = 0.5f;
    static const float MaxPlaybackRate = 2.0f;

    if (rate > MaxPlaybackRate) {
        rate = MaxPlaybackRate;
    }

    if (rate < MinPlaybackRate) {
        rate = MinPlaybackRate;
    }

    if (!m_audioQueue) {
        m_playbackRate = rate;
        return MediaResult::Ok;
    } else {
        OSStatus status = AudioQueueSetParameter(m_audioQueue, kAudioQueueParam_PlayRate, rate);
        return check(status, "AudioQueueSetParameter kAudioQueueParam_PlayRate");
    }
}

MediaResult AudioQueueRenderer::getRenderedPresentationTime(MediaTime& time)
{
    time = MediaTime::invalid();

    const bool UseAudioQueueTime = false;

    if (m_started) {
        if (UseAudioQueueTime) {
            OSStatus status = AudioQueueGetCurrentTime(m_audioQueue, NULL, &m_audioTimestamp, NULL);
            if (status == noErr) {
                if (m_audioTimestamp.mFlags & kAudioTimeStampSampleTimeValid) {
                    time = m_initialSampleTime + MediaTime(m_audioTimestamp.mSampleTime, m_description.mSampleRate);
                } else {
                    m_log->warn("AudioQueueGetCurrentTime returned invalid time");
                }
            } else if (status == kAudioQueueErr_InvalidRunState) {
                // normal queue may not be in running state yet
            } else {
                return check(status, "AudioQueueGetCurrentTime");
            }
        } else {
            time = m_lastPresentationTime;
        }
    }

    return MediaResult::Ok;
}

MediaResult AudioQueueRenderer::check(OSStatus status, const char* message)
{
    if (status != noErr) {
        m_log->error("error %d message %s", status, message);
        return MediaResult(MediaResult::Error, status);
    }

    return MediaResult::Ok;
}

void AudioQueueRenderer::outputCallback(void* inUserData, AudioQueueRef inAQ, AudioQueueBufferRef inBuffer)
{
    (void)inAQ;
    AudioQueueRenderer* renderer = reinterpret_cast<AudioQueueRenderer*>(inUserData);

    if (renderer) {
        {
            std::lock_guard<std::mutex> lock(renderer->m_mutex);
            renderer->fillBuffer(inBuffer);
        }
        renderer->m_bufferAvailable.notify_one();
    }
}

void AudioQueueRenderer::propertyListenerCallback(void* inUserData, AudioQueueRef inAQ, AudioQueuePropertyID inID)
{
    (void)inAQ;
    AudioQueueRenderer* renderer = reinterpret_cast<AudioQueueRenderer*>(inUserData);

    switch (inID) {
    case kAudioQueueProperty_IsRunning:
        if (renderer) {
            UInt32 data = 0;
            UInt32 size = sizeof(data);
            OSStatus status = AudioQueueGetProperty(inAQ, kAudioQueueProperty_IsRunning, &data, &size);
            if (status == noErr) {
                renderer->m_log->info("AudioQueueDecoder kAudioQueueProperty_IsRunning %s", data ? "YES" : "NO");
            }
            // TODO if not running and started?
        }
        break;
    default:
        break;
    }
}
}
