#import "CocoaHttpRequest.h"
#import "CocoaHttpResponse.h"

namespace twitch {
CocoaHttpRequest::CocoaHttpRequest(const std::string& url, HttpMethod method)
    : HttpRequest(url, method)
    , m_urlRequest(nullptr)
    , m_task(nullptr)
    , m_streamTask(nullptr)
    , m_cancelled(false)
{
    NSString* urlString = [NSString stringWithUTF8String:url.c_str()];
    NSURL* nsUrl = [NSURL URLWithString:urlString];
    m_urlRequest = [NSMutableURLRequest requestWithURL:nsUrl];
}

void CocoaHttpRequest::cancel()
{
    m_cancelled = true;
    [m_task cancel];
    [m_streamTask cancel];
}

void CocoaHttpRequest::setHeader(const std::string& key, const std::string& value)
{
    NSString* nsField = [NSString stringWithUTF8String:key.c_str()];
    NSString* nsValue = [NSString stringWithUTF8String:value.c_str()];
    [m_urlRequest addValue:nsValue forHTTPHeaderField:nsField];
}

void CocoaHttpRequest::setContent(const std::vector<uint8_t>& content)
{
    if (!content.empty()) {
        m_urlRequest.HTTPBody = [NSData dataWithBytes:content.data() length:content.size()];
    }
}

NSURLSessionTask* CocoaHttpRequest::send(NSURLSession* session, HttpClient::ResponseHandler onResponse, HttpClient::ErrorHandler onError)
{
    [m_urlRequest setTimeoutInterval:static_cast<double>(getTimeout().count())];
    [m_urlRequest setHTTPMethod:[NSString stringWithUTF8String:getMethodString().c_str()]];
    [m_urlRequest setCachePolicy:NSURLRequestReloadIgnoringCacheData];

    m_task = [session dataTaskWithRequest:m_urlRequest];
    m_response.reset();
    m_onResponse = onResponse;
    m_onError = onError;
    return m_task;
}

void CocoaHttpRequest::onCompletedWithError(NSError* error)
{
    const int ecode = error ? static_cast<int>(error.code) : 0;
    if (ecode == NSURLErrorCancelled) {
        return;
    }
    if (!m_response) {
        if (ecode) {
            m_onError(ecode);
        }
        return;
    }
    if (ecode) {
        if (m_response->getErrorHandler()) {
            m_response->getErrorHandler()(ecode);
        }
    } else {
        m_response->onData(nullptr, 0, true);
    }
}

void CocoaHttpRequest::onResponse(std::shared_ptr<CocoaHttpResponse> response)
{
    m_response = response;
    m_onResponse(m_response);
}
    
void CocoaHttpRequest::onData(const uint8_t* data, size_t size, bool endOfStream)
{
    m_response->onData(data, size, endOfStream);
}
}
