//
//  DisplayLayerRenderer.mm
//  player-core
//
//  Created by Purushe, Nikhil on 11/1/16.
//
//

#include "DisplayLayerRenderer.h"
#include "CMSampleBufferSample.hpp"
#include <Foundation/Foundation.h>
#include <thread>

namespace twitch {
DisplayLayerRenderer::DisplayLayerRenderer(const ReferenceClock& clock)
    : m_clock(clock)
    , m_playbackRate(1.0)
    , m_layer(nil)
    , m_lastPresentationTimeStamp(kCMTimeInvalid)
    , m_droppedFrames(0)
    , m_renderedFrames(0)
{
}

MediaResult DisplayLayerRenderer::configure(const MediaFormat& format)
{
    (void)format;
    return MediaResult::Ok;
}

MediaResult DisplayLayerRenderer::render(std::shared_ptr<const MediaSample> input)
{
    auto imageSample = std::static_pointer_cast<const CMSampleBufferSample>(input);
    if (!imageSample) {
        return MediaResult::ErrorInvalidData;
    }

    CMSampleBufferRef sampleBuffer = imageSample->ref;
    if (!sampleBuffer || !CMSampleBufferIsValid(sampleBuffer)) {
        NSLog(@"Sample Buffer Invalid");
        return MediaResult::ErrorInvalidData;
    }

    bool render = true;
    
    // wait until the layer can accept data, drop if it times out
    MediaTime start = MediaTime::now();
    while (m_layer && !m_layer.readyForMoreMediaData) {
        std::this_thread::sleep_for(input->duration.microseconds());
        
        if (MediaTime::now() - start > MediaTime(0.25)) {
            NSLog(@"AVSampleBufferDisplayerLayer readyForMoreMediaData timed out");
            render = false;
            break;
        }
    }

    MediaTime syncTime = m_clock.getMediaTime();

    if (syncTime.valid()) {
        // periodically resync to the master media time
        if (syncTime == MediaTime::zero() || !m_mediaTime.valid() || syncTime - m_mediaTime > (MediaTime(1.0) * m_playbackRate)) {
            m_mediaTime = syncTime;
            m_startTime = MediaTime::now();
            NSLog(@"AVSampleBufferDisplayerLayer set controlTimebase to %.3f", syncTime.seconds());
            CMTimebaseSetTime(m_layer.controlTimebase, CMTimeMake(syncTime.count(), syncTime.timebase()));
        }
        
        // check if this output pts is greater than the last since the frames aren't ordered by pts in b frame case
        if (CMTimeCompare(CMSampleBufferGetPresentationTimeStamp(sampleBuffer), m_lastPresentationTimeStamp) == 1) {
            static const MediaTime syncInterval = MediaTime(1 / 60.0);
            MediaTime elapsed = (MediaTime::now() - m_startTime) * m_playbackRate;
            MediaTime currentTime = m_mediaTime + elapsed;
            MediaTime releaseTime = currentTime > MediaTime::zero() ? input->presentationTime - currentTime : MediaTime::zero();
            
            if (releaseTime >= (syncInterval * -1) && m_startTime > MediaTime::zero()) {
                // normal case queue the buffer
            } else if (m_renderedFrames > 0) {
                //NSLog(@"release %lld ms", releaseTime.milliseconds().count());
                render = false;
            }
        }
    }

    MediaResult result = MediaResult::Ok;

    if (render) {
        [CATransaction begin];
        if ([m_layer status] == AVQueuedSampleBufferRenderingStatusFailed) {
            NSError* error = [m_layer error];
            NSLog(@"AVQueuedSampleBufferRenderingStatusFailed %@", error);
            result = MediaResult::Error;
        } else {
            [m_layer enqueueSampleBuffer:sampleBuffer];
            m_lastPresentationTimeStamp = CMSampleBufferGetPresentationTimeStamp(sampleBuffer);
            m_renderedFrames++;
        }
        [CATransaction commit];
    } else {
        m_droppedFrames++;
    }

    CFRelease(imageSample->ref);
    return result;
}

MediaResult DisplayLayerRenderer::flush()
{
    [CATransaction begin];
    [m_layer flush];
    [CATransaction commit];
    m_droppedFrames = 0;
    m_renderedFrames = 0;
    m_lastPresentationTimeStamp = kCMTimeInvalid;
    m_startTime = MediaTime::invalid();
    m_mediaTime = MediaTime::invalid();
    return MediaResult::Ok;
}

MediaResult DisplayLayerRenderer::start()
{
    m_startTime = MediaTime::now();
    return MediaResult::Ok;
}

MediaResult DisplayLayerRenderer::stop()
{
    m_startTime = MediaTime::invalid();
    m_mediaTime = MediaTime::invalid();
    return MediaResult::Ok;
}

MediaResult DisplayLayerRenderer::getDroppedFrames(int& count)
{
    count = m_droppedFrames;
    return MediaResult::Ok;
}

MediaResult DisplayLayerRenderer::getRenderedFrames(int& count)
{
    count = m_renderedFrames;
    return MediaResult::Ok;
}

MediaResult DisplayLayerRenderer::getRenderedPresentationTime(MediaTime& time)
{
    if (CMTIME_IS_INVALID(m_lastPresentationTimeStamp)) {
        time = MediaTime::invalid();
    } else {
        time = MediaTime(m_lastPresentationTimeStamp.value, m_lastPresentationTimeStamp.timescale);
    }
    return MediaResult::Ok;
}

MediaResult DisplayLayerRenderer::setPlaybackRate(float rate)
{
    MediaResult result = MediaResult::Ok;

    m_playbackRate = rate;
    [CATransaction begin];
    if (m_layer.controlTimebase) {
        CMTimebaseSetRate(m_layer.controlTimebase, rate);
    } else {
        result = MediaResult::ErrorNotSupported;
    }
    [CATransaction commit];

    return result;
}

MediaResult DisplayLayerRenderer::setSurface(void* surface)
{
    [CATransaction begin];
    [m_layer flushAndRemoveImage];
    AVSampleBufferDisplayLayer* layer = (__bridge AVSampleBufferDisplayLayer*)(surface);
    m_layer = layer;
    if (m_layer != nil) {
        [m_layer flushAndRemoveImage]; // if the view is being reused clear the image
        if (!m_layer.controlTimebase) {
            CMTimebaseRef controlTimebase;
            CMTimebaseCreateWithMasterClock(kCFAllocatorDefault, CMClockGetHostTimeClock(), &controlTimebase);
            CMTimebaseSetRate(controlTimebase, m_playbackRate);
            m_layer.controlTimebase = controlTimebase;
        }
    }
    [CATransaction commit];
    return MediaResult::Ok;
}
}
