
#import "TTVAVPlayerObserver.h"
#import "AVPlayerSink.h"
#import <AVFoundation/AVFoundation.h>

static int KVOContext = 0;

@implementation TTVAVPlayerObserver {
    id<NSObject> timeObserverToken;
    CMTime seekTime;
    AVPlayer* player;
    twitch::MediaSink::Listener* listener;
    std::shared_ptr<twitch::ScopedScheduler> scopedScheduler;
    std::shared_ptr<twitch::MediaFormat> format;
    std::deque<std::shared_ptr<twitch::MediaFormat>> mediaFormats;
    twitch::AVPlayerSink* playerSink;
    std::deque<std::shared_ptr<twitch::MediaSampleBuffer>> samples;
}

- (TTVAVPlayerObserver*)initWithPlayer:(AVPlayer*)player
                                      :(twitch::MediaSink::Listener*)listener
                                      :(std::shared_ptr<twitch::Scheduler>)scheduler
                                      :(twitch::AVPlayerSink*)AVplayerSink
{
    self = [super init];
    self->player = player;
    self->playerSink = AVplayerSink;
    self->listener = listener;
    self->scopedScheduler = std::shared_ptr<twitch::ScopedScheduler>(new twitch::ScopedScheduler(scheduler));

    //Watch Notifications
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(itemDidFinishPlaying:) name:AVPlayerItemDidPlayToEndTimeNotification object:nil];
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(itemDidNotFinishPlaying:) name:AVPlayerItemFailedToPlayToEndTimeNotification object:nil];
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(itemPlaybackStalled:) name:AVPlayerItemPlaybackStalledNotification object:nil];
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(handleRouteChange:) name:AVAudioSessionRouteChangeNotification object:nil];
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(newErrorLogEntry:) name:AVPlayerItemNewErrorLogEntryNotification object:nil];

    if (player.currentItem) {
        [self observeItem:player.currentItem];
    }
    [player addObserver:self forKeyPath:@"currentItem" options:NSKeyValueObservingOptionNew | NSKeyValueObservingOptionOld context:&KVOContext];
    [player addObserver:self forKeyPath:@"timeControlStatus" options:0 context:&KVOContext];

    TTVAVPlayerObserver __weak* weakSelf = self;
    timeObserverToken = [player addPeriodicTimeObserverForInterval:CMTimeMakeWithSeconds(0.1, NSEC_PER_SEC)
                                                             queue:dispatch_get_main_queue()
                                                        usingBlock:^(CMTime time) {
                                                            [weakSelf timeObserved:time];
                                                        }];
    return self;
}

- (void)dealloc
{
    // Remove Notifications
    [[NSNotificationCenter defaultCenter] removeObserver:self name:AVPlayerItemDidPlayToEndTimeNotification object:nil];
    [[NSNotificationCenter defaultCenter] removeObserver:self name:AVPlayerItemFailedToPlayToEndTimeNotification object:nil];
    [[NSNotificationCenter defaultCenter] removeObserver:self name:AVPlayerItemPlaybackStalledNotification object:nil];
    [[NSNotificationCenter defaultCenter] removeObserver:self name:AVAudioSessionRouteChangeNotification object:nil];
    [[NSNotificationCenter defaultCenter] removeObserver:self name:AVPlayerItemNewErrorLogEntryNotification object:nil];

    // Remove Observers
    [self unobserveItem:player.currentItem];
    [player removeObserver:self forKeyPath:@"currentItem" context:&KVOContext];
    [player removeObserver:self forKeyPath:@"timeControlStatus" context:&KVOContext];

    if (timeObserverToken) {
        [player removeTimeObserver:timeObserverToken];
        timeObserverToken = nil;
    }
    scopedScheduler->cancel();
}

- (void)seekTo:(CMTime)time
{
    seekTime = time;
}

- (void)observeItem:(AVPlayerItem*)item
{
    [item addObserver:self forKeyPath:@"status" options:NSKeyValueObservingOptionNew | NSKeyValueObservingOptionInitial context:&KVOContext];
    [item addObserver:self forKeyPath:@"presentationSize" options:NSKeyValueObservingOptionNew | NSKeyValueObservingOptionOld context:&KVOContext];
    [item addObserver:self forKeyPath:@"timedMetadata" options:NSKeyValueObservingOptionNew | NSKeyValueObservingOptionInitial context:&KVOContext];
    [item addObserver:self forKeyPath:@"playbackLikelyToKeepUp" options:NSKeyValueObservingOptionNew | NSKeyValueObservingOptionInitial context:&KVOContext];
    [item addObserver:self forKeyPath:@"playbackBufferEmpty" options:NSKeyValueObservingOptionNew | NSKeyValueObservingOptionInitial context:&KVOContext];
}

- (void)unobserveItem:(AVPlayerItem*)item
{
    [item removeObserver:self forKeyPath:@"status" context:&KVOContext];
    [item removeObserver:self forKeyPath:@"presentationSize" context:&KVOContext];
    [item removeObserver:self forKeyPath:@"timedMetadata" context:&KVOContext];
    [item removeObserver:self forKeyPath:@"playbackLikelyToKeepUp" context:&KVOContext];
    [item removeObserver:self forKeyPath:@"playbackBufferEmpty" context:&KVOContext];
}

- (CMTime)availableDuration
{
    NSValue *range = player.currentItem.loadedTimeRanges.firstObject;
    if (range != nil){
        return CMTimeRangeGetEnd(range.CMTimeRangeValue);
    }
    return kCMTimeZero;
}

- (void)timeObserved:(CMTime)time
{
    twitch::MediaTime currentTime(time.value, time.timescale);
    twitch::MediaTime baseTime = twitch::MediaTime(seekTime.value, seekTime.timescale);
    // update for seekTime
    if (baseTime != twitch::MediaTime::zero()) {
        currentTime += baseTime;
    }

    // onSinkTimeUpdate
    listener->onSinkTimeUpdate(currentTime);

    // onSinkMetadataSample
    if (format && !samples.empty()) {
        scopedScheduler->schedule([=]() {
            for (const auto& sample : samples) {
                if (sample->presentationTime > currentTime) {
                    listener->onSinkMetadataSample(*sample);
                }
            }
            samples.clear();
        });
    }
}

- (void)itemPlaybackStalled:(NSNotification*)notification
{
    (void)notification;
    NSLog(@"Item playback stalled");
}

- (void)handleRouteChange:(NSNotification*)notification
{
    (void)notification;
    //    NSLog(@"Route change notification");
    // Reset is not needed for passthrough mode
    scopedScheduler->schedule([=]() {
        if (player.allowsExternalPlayback && !playerSink->isPassthroughMode()) {
            playerSink->reset();
        }
    });
}

- (void)newErrorLogEntry:(NSNotification*)notification
{
    (void)notification;
    //AVPlayerItemErrorLog* errorLog = [player.currentItem errorLog];
    //NSLog(@"AVPlayerItemErrorLog: %@", errorLog);
}

- (void)enqueueFormat:(std::shared_ptr<twitch::MediaFormat>)format
{
    if (self->format != format) {
        mediaFormats.emplace_back(format);
        self->format = format;
    }
}

- (void)enqueueSample:(std::shared_ptr<twitch::MediaSampleBuffer>)sample
{
    if (sample) {
        samples.emplace_back(sample);
    }
}

- (void)itemDidFinishPlaying:(NSNotification*)notification
{
    (void)notification;

    //handle endOfStream
    NSLog(@"Item finished Playing");
    scopedScheduler->schedule([=]() {
        listener->onSinkStateChanged(twitch::MediaSink::State::Idle);
    });
}

- (void)itemDidNotFinishPlaying:(NSNotification*)notification
{
    (void)notification;
    NSLog(@"Item failed to finish Playing");
}

//KV Observation
- (void)observeValueForKeyPath:(NSString*)keyPath ofObject:(id)object change:(NSDictionary*)change context:(void*)context
{
    if (context != &KVOContext) {
        // KVO isn't for us.
        [super observeValueForKeyPath:keyPath ofObject:object change:change context:context];
        return;
    }

    if ([keyPath isEqualToString:@"timeControlStatus"]) {
        if (player.timeControlStatus == AVPlayerTimeControlStatusPlaying) {
            scopedScheduler->schedule([=]() {
                listener->onSinkStateChanged(twitch::MediaSink::State::Playing);
            });
        }
    } else if ([keyPath isEqualToString:@"currentItem"]) {
        if (change[NSKeyValueChangeOldKey] != [NSNull null]) {
            AVPlayerItem* old = change[NSKeyValueChangeOldKey];
            [self unobserveItem:old];
        }
        if (change[NSKeyValueChangeNewKey] != [NSNull null]) {
            AVPlayerItem* current = change[NSKeyValueChangeNewKey];
            [self observeItem:current];
        }
    } else if ([keyPath isEqualToString:@"status"]) {
        // Display an error if status becomes Failed
        // Handle NSNull value for NSKeyValueChangeNewKey, i.e. when player.currentItem is nil
        NSNumber* newStatusAsNumber = change[NSKeyValueChangeNewKey];
        AVPlayerItemStatus newStatus = [newStatusAsNumber isKindOfClass:[NSNumber class]] ? AVPlayerItemStatus([newStatusAsNumber integerValue]) : AVPlayerItemStatusUnknown;

        if (newStatus == AVPlayerItemStatusFailed) {
            NSLog(@"Error occured with message: %@, error: %@.", player.currentItem.error.localizedDescription, player.currentItem.error);

            const std::string message = std::string([player.currentItem.error.localizedDescription UTF8String]);
            twitch::Error error(twitch::ErrorSource::Render, twitch::MediaResult::Error, message);

            //handle error
            scopedScheduler->schedule([=]() {
                listener->onSinkError(error);
            });
        }
    } else if ([keyPath isEqualToString:@"playbackLikelyToKeepUp"]) {

        if (player.currentItem.playbackLikelyToKeepUp) {
            NSLog(@"Item playbackLikelyToKeepUp");
            [player play];
        }
    } else if ([keyPath isEqualToString:@"playbackBufferEmpty"]) {

        if (player.currentItem.playbackBufferEmpty) {
            NSLog(@"Buffer empty current position is %f, end is %f, range is %f", CMTimeGetSeconds(player.currentTime), CMTimeGetSeconds([self availableDuration]), CMTimeGetSeconds([self availableDuration]) - CMTimeGetSeconds(player.currentTime));
            scopedScheduler->schedule([=]() {
                listener->onSinkStateChanged(twitch::MediaSink::State::Idle);
            });
        }
    } else if ([keyPath isEqualToString:@"presentationSize"]) {

        CGSize newSize = CGSize([change[NSKeyValueChangeNewKey] CGSizeValue]);
        CGSize oldSize = CGSize([change[NSKeyValueChangeOldKey] CGSizeValue]);

        if (!CGSizeEqualToSize(newSize, oldSize)) {
            NSLog(@"Size changed from old size :(%f,%f) to new size :(%f,%f)", oldSize.width, oldSize.height, newSize.width, newSize.height);

            if (!mediaFormats.empty()) {
                scopedScheduler->schedule([=]() {
                    for (const auto& buffer : mediaFormats) {
                        NSLog(@"Format changed");
                        listener->onSinkFormatChanged(*buffer);
                    }
                    mediaFormats.clear();
                });
            }
        }

    } else if ([keyPath isEqualToString:@"timedMetadata"]) {
        id newMetadataArray = [change objectForKey:NSKeyValueChangeNewKey];
        if (newMetadataArray != [NSNull null] && [newMetadataArray count] > 0) {
            NSMutableArray* arr = [[NSMutableArray alloc] init];

            //Parse metadata and add to array
            for (AVMetadataItem* metadataItem in newMetadataArray) {
                NSDictionary* jsondict = [self parseMetadata:metadataItem];
                [arr addObject:jsondict];
            }

            //Create new dictionary with key ID3 and value as array(parsed from metadata)
            NSDictionary* dict = [NSDictionary dictionaryWithObjectsAndKeys:arr, @"ID3", nil];

            //Convert to Json string
            NSError* error;
            NSData* jsonData = [NSJSONSerialization dataWithJSONObject:dict options:0 error:&error];
            NSString* jsonString = [[NSString alloc] initWithData:jsonData encoding:NSUTF8StringEncoding];
            std::string content = std::string([jsonString UTF8String]);

            //create Metadata sample for callback
            auto position = player.currentItem.currentTime;
            twitch::MediaTime timeStamp(position.value, position.timescale);
            [self createMetadataSample:content:timeStamp];
        }
    } else {
        [super observeValueForKeyPath:keyPath ofObject:object change:change context:context];
    }
}

- (void)createMetadataSample:(const std::string&)content
                            :(twitch::MediaTime)timestamp
{
    auto sample = std::make_shared<twitch::MediaSampleBuffer>();
    sample->decodeTime = timestamp;
    sample->presentationTime = timestamp;
    sample->buffer = std::vector<uint8_t>(content.begin(), content.end());

    scopedScheduler->schedule([=]() {
        listener->onSinkMetadataSample(*sample);
    });
}

- (NSDictionary*)parseMetadata:(AVMetadataItem*)metadataItem
{
    id extraAttributeType = [metadataItem extraAttributes];
    NSString* extraString = nil;
    if ([extraAttributeType isKindOfClass:[NSDictionary class]]) {
        extraString = [extraAttributeType valueForKey:@"info"];
    } else if ([extraAttributeType isKindOfClass:[NSString class]]) {
        extraString = extraAttributeType;
    }

    id key = [metadataItem key];
    id value = [metadataItem value];

    NSDictionary* jsonDictionary;
    if ([key isEqual:@"TXXX"]) {
        NSArray* valueArr = [NSArray arrayWithObject:value];
        jsonDictionary = [NSDictionary dictionaryWithObjectsAndKeys:key, @"id", extraString, @"desc", valueArr, @"info", nil];
    } else if ([key isEqual:@"TSSE"] || [key isEqual:@"TDEN"] || [key isEqual:@"TDTG"]
        || [key isEqual:@"TOFN"] || [key isEqual:@"TRCK"]) {
        jsonDictionary = [NSDictionary dictionaryWithObjectsAndKeys:key, @"id", value, @"info", nil];
    } else {
        NSLog(@"Unhadeled id3 key '%@' and value %@", key, value);
    }

    return jsonDictionary;
}

@end
