#import "TTVPlayerDelegate.h"
#import "TTVPlayerLayer.h"
#import <AVFoundation/AVFoundation.h>
#import <CoreMedia/CoreMedia.h>
#import <Foundation/Foundation.h>

NS_ASSUME_NONNULL_BEGIN

/*!
    @enum TTVPlayerState
    @abstract
        These constants are returned by the player to indicates it's current state.
    @constant	 TTVPlayerStateIdle
        Indicates that the status of the player is idle
    @constant	 TTVPlayerStateReady
        Indicates that the player is ready to play the selected source
    @constant	 TTVPlayerStateBuffering
        Indicates that the player is buffering content
    @constant	 TTVPlayerStatePlaying
        Indicates the player is playing
    @constant	 TTVPlayerStateEnded
        Indicates that the player has reached the end of the stream
 */
typedef NS_ENUM(NSInteger, TTVPlayerState) {
    TTVPlayerStateIdle,
    TTVPlayerStateReady,
    TTVPlayerStateBuffering,
    TTVPlayerStatePlaying,
    TTVPlayerStateEnded
};

@class TTVQuality;
@protocol TTVPlayerDelegate;

/*!
	@class			TTVPlayer
	@abstract Player interface for the twitch player
 */
@interface TTVPlayer : NSObject

- (instancetype)init NS_UNAVAILABLE;

/*!
 @method          initWithLayer:
 @abstract        Initializes an TTVPlayer that renders to the given layer.
 @param           layer
 @result          An instance of TTVPlayer
 */
- (instancetype _Nonnull)initWithLayer:(CALayer* _Nullable)layer;

/*!
    @property	delegate
    @abstract	Sets the delegate to receive player state changes and updates
 */
@property (nonatomic, weak, nullable) id<TTVPlayerDelegate> delegate;

/*!
 @property    layer
 @abstract    Gets the layer used for video playback
 */
@property (nonatomic, nullable) CALayer* layer;

/*!
    @property	autoSwitchQuality
    @abstract	Indicates whether the player should switch qualities to the best bandwdith.
 */
@property (nonatomic) BOOL autoSwitchQuality;

/*!
    @property	looping
    @abstract	Indicates whether the player should loop the content, defaults to false.
 */
@property (nonatomic) BOOL looping;

/*!
 @property    liveLatency
 @abstract    Indicates the current source latency for live streams.
 */
@property (nonatomic, readonly) CMTime liveLatency;

/*!
    @property	muted
    @abstract	Indicates the muted state of the player, defaults to false.
 */
@property (nonatomic) BOOL muted;

/*!
    @property	playbackRate
    @abstract	Indicates the desired rate of playback rate.
 */
@property (nonatomic) float playbackRate;

/*!
 @property    paused
 @abstract    Indicates if the player is paused.
 */
@property(nonatomic, readonly) BOOL paused;

/*!
    @property	volume
    @abstract	Sets the volume of the audio track of the player, defaults to 1.0
 */
@property (nonatomic) float volume;

/*!
    @property   bandwidthEstimate
    @abstract   Indicates the current player's bandwdith estimate in bps.
 */
@property (nonatomic, readonly) long bandwidthEstimate;

/*!
    @property	buffered
    @abstract	Indicates the current buffer end position.
 */
@property (nonatomic, readonly) CMTime buffered;

/*!
    @property	duration
    @abstract	Indicates the current duration of the playing media.
 */
@property (nonatomic, readonly) CMTime duration;

/*!
    @property	position
    @abstract	Indicates the current playing position.
 */
@property (nonatomic, readonly) CMTime position;

/*!
    @property	seekable
    @abstract	Indicates whether the current media can be seeked to with seekTo
 */
@property (nonatomic, readonly) BOOL seekable;

/*!
    @property	path
    @abstract	Indicates the current path or url the player has loaded
 */
@property (nonatomic, readonly, nonnull) NSURL* path;

/*!
    @property    qualitiy
    @abstract    Indicates the current quality being played.
 */
@property (nonatomic, readonly, nonnull) TTVQuality* quality;

/*!
    @property	qualitiesSet
    @abstract	Indicates the current set of available qualities.
 */
@property (nonatomic, readonly, nonnull) NSSet* qualities;

/*!
    @property	version
    @abstract	Indicates the current version of the player.
 */
@property (nonatomic, readonly, nonnull) NSString* version;

/*!
    @property   videoBitrate
    @abstract   Indicates the current player's currently playing video bitrate.
 */
@property (nonatomic, readonly) long videoBitrate;

/*!
    @property    videoFramesDecoded
    @abstract    Indicates the current number of decoded frames
 */
@property (nonatomic, readonly) int videoFramesDecoded;

/*!
    @property   videoFramesDecoded
    @abstract   Indicates the current number of dropped frames
 */
@property (nonatomic, readonly) int videoFramesDropped;

/*!
    @property	state
    @abstract	Indicates the current state of the player.
 */
@property (nonatomic, readonly) TTVPlayerState state;

/*!
 @property    allowsExternalPlayback
 @abstract    Indicates whether the player allows switching to external playback mode.
 */
@property (nonatomic) BOOL allowsExternalPlayback;

/*!
 @property    externalPlaybackActive
 @abstract    Indicates whether the player is playing video in external playback mode.
 */
@property (nonatomic, readonly) BOOL externalPlaybackActive;

/*!
 @property    usesExternalPlaybackWhileExternalScreenIsActive
 @abstract    Indicates whether the player should automatically switch
              to external playback mode while the external screen mode is active.
 */
@property (nonatomic) BOOL usesExternalPlaybackWhileExternalScreenIsActive;

/*!
 @property    naturalSize
 @abstract    Indicates the natural size of the player.
 */
@property (nonatomic, readonly) CGSize naturalSize;

/*!
 @property    currentItem
 @abstract    Indicates the player's current player item.
 */
@property (nonatomic, readonly) AVPlayerItem* currentItem;

/*!
 @property   avPlayer
 @abstract   Indicates the current AVPlayer instance.
 */
@property (nonatomic, readonly) AVPlayer* avPlayer;

/*!
    @method     load:
    @abstract	loads a media url
    @param		path path to streaming manifest, clip or file for playback.
 */
- (void)load:(NSURL* _Nullable)path;

/*!
    @method     pause
    @abstract	Pauses playback after which the player will transition to the IDLE state
 */
- (void)pause;

/*!
    @method     play
    @abstract	begins playback if the in ready state otherwise playback is started after entering the Ready state.
 */
- (void)play;

/*!
    @method     seekTo:
    @abstract	seeks to the given position.
    @param		position
 */
- (void)seekTo:(CMTime)position;

/*!
 @method     seekTo:
 @abstract   seeks to the given position and invokes completion handler.
 @param      position
 */
- (void)seekTo:(CMTime)position
    completionHandler:(void (^)(BOOL finished))completionHandler;
/*!
    @method     setQualityByName:
    @abstract   selects the given quality
    @param      quality
 */
- (void)setQualityByName:(NSString*)quality;

/*!
 @method    setLiveLowLatency:
 @abstract  sets live low latency
 @param     enable
 */
- (void)setLiveLowLatency:(BOOL)enable;

/*!
 @method    addPeriodicTimeObserverForInterval:
 @abstract	Requests to invoke the *block* at playbacktime *interval*.
 @param		interval playback timeline interval
 */
- (id)addPeriodicTimeObserverForInterval:(CMTime)interval
                                   queue:(dispatch_queue_t _Nullable)queue
                              usingBlock:(void (^_Nonnull)(CMTime time))block;

/*!
 @method    addBoundaryTimeObserverForTimes:
 @abstract	Requests to invoke the *block* at designated playback *times*.
 @param		times playback timeline boundaries
 */
- (id)addBoundaryTimeObserverForTimes:(NSArray<NSValue*>* _Nullable)times
                                queue:(dispatch_queue_t _Nullable)queue
                           usingBlock:(void (^_Nonnull)(void))block;

/*!
 @method    removeTimeObserver:
 @abstract	remove playback time observer.
 @param		observer
 */
- (void)removeTimeObserver:(id)observer;

/*!
 @method    cancelPendingPrerolls:
 @abstract  cancel AVPlayer pending prerolls.
 */
- (void)cancelPendingPrerolls;

/*!
 @property    closedCaptionDisplayEnabled
 @abstract    Indicates if closed caption are enabled, defaults to false.
 */

@property (nonatomic) BOOL closedCaptionDisplayEnabled;

@end

NS_ASSUME_NONNULL_END
