//
//  TTVPlayer.mm
//  player-core
//
//  Created by Purushe, Nikhil on 3/31/17.
//
//

#import "TTVPlayer.h"
#import "AVPlayerSink.h"
#import "ApplePlatform.h"
#import "TTVObserver.h"
#import "TTVQuality.h"
#import "player/MediaPlayer.hpp"

using namespace twitch;
struct PlayerListener;

@implementation TTVPlayer {
    TTVQuality* _quality;
    std::unique_ptr<PlayerListener> m_listener;
    std::unique_ptr<MediaPlayer> m_player;
    std::shared_ptr<ApplePlatform> m_platform;
    void (^_Nullable completionHandler)(BOOL);
    AVPlayer* _avplayer;
    CALayer* _layer;
}

- (instancetype)initWithLayer:(CALayer*)layer
{
    _layer = layer;
    m_platform = std::make_shared<ApplePlatform>();
    m_platform->setPassthroughMode([layer isKindOfClass:[AVPlayerLayer class]]);
    m_listener.reset(new PlayerListener(self));
    m_player.reset(new MediaPlayer(*m_listener, m_platform));
    m_player->setSurface((__bridge void*)layer);
    return self;
}

- (void)dealloc
{
    [self removeObservers:_avplayer];
    m_player.reset();
}

- (void)addObservers:(AVPlayer*)player
{
    [player addObserver:self forKeyPath:@"externalPlaybackActive" options:NSKeyValueObservingOptionNew | NSKeyValueObservingOptionOld context:nil];
    [player addObserver:self forKeyPath:@"currentItem" options:0 context:nil];
}

- (void)removeObservers:(AVPlayer*)player
{
    [player removeObserver:self forKeyPath:@"externalPlaybackActive" context:nil];
    [player removeObserver:self forKeyPath:@"currentItem" context:nil];
}

- (void)observeValueForKeyPath:(NSString*)keyPath ofObject:(id)object change:(NSDictionary*)change context:(void*)context
{
    (void)object;
    (void)change;
    (void)context;
    TTVPlayer* __weak weakSelf = self;
    [[NSOperationQueue mainQueue] addOperationWithBlock:^(void) {
        if ([keyPath isEqualToString:@"externalPlaybackActive"]) {
            [weakSelf.delegate onExternalPlaybackChanged:weakSelf];
        } else if ([keyPath isEqualToString:@"currentItem"]) {
            [weakSelf.delegate onPlayerItemChanged:weakSelf:weakSelf.currentItem];
        }
    }];
}

- (BOOL)paused { return (self.state == TTVPlayerStateIdle); }
- (BOOL)autoSwitchQuality { return m_player->getAutoSwitchQuality(); }
- (void)setAutoSwitchQuality:(BOOL)enable { m_player->setAutoSwitchQuality(enable); }

- (BOOL)closedCaptionDisplayEnabled
{
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdeprecated-declarations"
    return self.avPlayer.closedCaptionDisplayEnabled;
#pragma clang diagnostic pop
}

- (void)setClosedCaptionDisplayEnabled:(BOOL)enable
{
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wdeprecated-declarations"
    self.avPlayer.closedCaptionDisplayEnabled = enable;
#pragma clang diagnostic pop
}

- (void)setLiveLowLatency:(BOOL)enable
{
    m_player->setLiveLowLatencyEnabled(enable);
}

- (CALayer*)layer
{
    return _layer;
}

- (void)setLayer:(CALayer*)layer
{
    _layer = layer;
    m_player->setSurface((__bridge void*)layer);
}

- (BOOL)looping { return m_player->isLooping(); }
- (void)setLooping:(BOOL)enable { m_player->setLooping(enable); }

- (BOOL)muted { return m_player->isMuted(); }
- (void)setMuted:(BOOL)mute { m_player->setMuted(mute); }

- (float)playbackRate { return m_player->getPlaybackRate(); }
- (void)setPlaybackRate:(float)rate { m_player->setPlaybackRate(rate); }

- (float)volume { return m_player->getVolume(); }
- (void)setVolume:(float)volume { m_player->setVolume(volume); }

- (CGSize)naturalSize
{
    if (self.avPlayer != nil && self.avPlayer.currentItem != nil) {
        for (AVPlayerItemTrack* _Nonnull track in self.avPlayer.currentItem.tracks) {
            AVAssetTrack* _Nonnull assetTrack = track.assetTrack;
            if ([assetTrack.mediaType isEqualToString:AVMediaTypeVideo]) {
                return [assetTrack naturalSize];
            }
        }
    }
    return CGSizeZero;
}

- (BOOL)allowsExternalPlayback
{
    return self.avPlayer.allowsExternalPlayback;
}

- (void)setAllowsExternalPlayback:(BOOL)enable
{
    self.avPlayer.allowsExternalPlayback = enable;
}

- (BOOL)externalPlaybackActive
{
    return self.avPlayer.isExternalPlaybackActive;
}

- (BOOL)usesExternalPlaybackWhileExternalScreenIsActive
{
    return self.avPlayer.usesExternalPlaybackWhileExternalScreenIsActive;
}

- (void)setUsesExternalPlaybackWhileExternalScreenIsActive:(BOOL)enable
{
    self.avPlayer.usesExternalPlaybackWhileExternalScreenIsActive = enable;
}

- (long)bandwidthEstimate
{
    return m_player->getBandwidthEstimate();
}

- (CMTime)buffered
{
    MediaTime buffered = m_player->getBufferedPosition();
    return CMTimeMake(buffered.count(), buffered.timebase());
}

- (CMTime)position
{
    MediaTime position = m_player->getPosition();
    return CMTimeMake(position.count(), position.timebase());
}

- (CMTime)duration
{
    MediaTime duration = m_player->getDuration();
    if (duration == MediaTime::max()) {
        return kCMTimeIndefinite;
    } else {
        return CMTimeMake(duration.count(), duration.timebase());
    }
}

- (CMTime)liveLatency
{
    MediaTime latency = m_player->getLiveLatency();
    return CMTimeMake(latency.count(), latency.timebase());
}

- (BOOL)seekable { return m_player->isSeekable(); }

- (NSURL*)path
{
    return [NSURL URLWithString:[NSString stringWithUTF8String:m_player->getPath().c_str()]];
}

- (TTVQuality*)quality
{
    if (!_quality) {
        const auto& current = m_player->getQuality();
        NSString* name = [NSString stringWithUTF8String:current.name.c_str()];
        NSString* group = [NSString stringWithUTF8String:current.group.c_str()];
        _quality = [[TTVQuality alloc] initWithValues:name:group:current.bitrate:current.width:current.height];
    }
    return _quality;
}

- (NSSet*)qualities
{
    NSMutableSet* set = [NSMutableSet set];
    for (const Quality& quality : m_player->getQualities()) {
        NSString* name = [NSString stringWithUTF8String:quality.name.c_str()];
        NSString* group = [NSString stringWithUTF8String:quality.group.c_str()];
        [set addObject:[[TTVQuality alloc] initWithValues:name:group:quality.bitrate:quality.width:quality.height]];
    }
    return set;
}

- (TTVPlayerState)state { return (TTVPlayerState)m_player->getState(); }

- (NSString*)version
{
    return [NSString stringWithUTF8String:m_player->getVersion().c_str()];
}

- (long)videoBitrate
{
    return m_player->getStatistics().getBitRate();
}

- (int)videoFramesDecoded
{
    return m_player->getStatistics().getDecodedFrames();
}

- (int)videoFramesDropped
{
    return m_player->getStatistics().getDroppedFrames();
}

- (void)load:(NSURL*)path
{
    if (path == nil) {
        [self.avPlayer replaceCurrentItemWithPlayerItem:nil];
        return;
    }
    m_player->load(std::string([path.absoluteString UTF8String]));
}

- (void)pause
{
    m_player->pause();
}

- (void)play
{
    m_listener->resetObservers();
    m_player->play();
}

- (void)seekTo:(CMTime)position
{
    m_player->seekTo(MediaTime(position.value, position.timescale));
}

- (void)seekTo:(CMTime)position
    completionHandler:(void (^)(BOOL finished))completionHandler
{
    m_player->seekTo(MediaTime(position.value, position.timescale));
    if (self->completionHandler) {
        completionHandler(NO);
        self->completionHandler = nil;
    } else {
        self->completionHandler = [completionHandler copy];
    }
}

- (void)setQualityByName:(NSString*)name
{
    for (const Quality& quality : m_player->getQualities()) {
        NSString* qualityName = [NSString stringWithUTF8String:quality.name.c_str()];
        if ([name caseInsensitiveCompare:qualityName] == NSOrderedSame) {
            m_player->setQuality(quality);
        }
    }
}

- (id)addPeriodicTimeObserverForInterval:(CMTime)interval
                                   queue:(dispatch_queue_t _Nullable)queue
                              usingBlock:(void (^_Nonnull)(CMTime time))block
{
    TTVPeriodicObserver* to = [[TTVPeriodicObserver alloc] init];
    to.interval = interval;
    to.queue = queue;
    to.block = block;
    m_listener->addObserver(to);
    return to;
}

- (id)addBoundaryTimeObserverForTimes:(NSArray<NSValue*>* _Nullable)times
                                queue:(dispatch_queue_t _Nullable)queue
                           usingBlock:(void (^_Nonnull)(void))block
{
    TTVBoundaryObserver* to = [[TTVBoundaryObserver alloc] initWithValues:[times count]];
    to.times = times;
    to.queue = queue;
    to.block = block;
    m_listener->addObserver(to);
    return to;
}

- (void)removeTimeObserver:(id _Nonnull)observer
{
    m_listener->removeObserver(observer);
}

- (void)cancelPendingPrerolls
{
    [self.avPlayer cancelPendingPrerolls];
}

- (AVPlayer*)avPlayer
{
    if ([_layer isKindOfClass:[AVPlayerLayer class]]) {
        AVPlayer* player = static_cast<AVPlayerSink&>(m_player->getSink()).getPlayer();
        if (_avplayer != player) {
            [self removeObservers:_avplayer];
            [self addObservers:player];
            _avplayer = player;
        }
    }
    return _avplayer;
}

- (AVPlayerItem*)currentItem
{
    return self.avPlayer.currentItem;
}

struct PlayerListener : public Player::Listener {
    TTVPlayer* __weak m_player;
    NSMutableSet* m_observers;

    PlayerListener(TTVPlayer* player)
        : m_player(player)
    {
        m_observers = [[NSMutableSet alloc] init];
    }

    ~PlayerListener()
    {
        m_player = nil;
        m_observers = nil;
    }

    void onDurationChanged(MediaTime duration) override
    {
        (void)duration;
        TTVPlayer* __weak weakSelf = m_player;
        [[NSOperationQueue mainQueue] addOperationWithBlock:^(void) {
            if (weakSelf.delegate) {
                [weakSelf.delegate onDurationChanged:weakSelf];
            }
        }];
    }

    void onError(const Error& error) override
    {
        TTVPlayer* __weak weakSelf = m_player;
        NSString* errorMessage = [NSString stringWithUTF8String:error.message.c_str()];
        NSErrorDomain domain = @"MediaPlayer";
        int code = error.result.value;
        if (error.result.code != 0) {
            switch (error.source) {
            // expect all decode/render error codes to be OSStatus errors
            case ErrorSource::Decode:
            case ErrorSource::Render:
                domain = NSOSStatusErrorDomain;
                code = error.result.code;
                break;
            default:
                break;
            }
        }

        NSDictionary* info = @{
            NSLocalizedDescriptionKey : errorMessage,
            @"code" : [NSNumber numberWithInteger:error.result.value],
            @"source" : [NSString stringWithUTF8String:errorSourceString(error.source)]
        };
        NSError* nsError = [NSError errorWithDomain:domain code:code userInfo:info];
        [[NSOperationQueue mainQueue] addOperationWithBlock:^(void) {
            if (weakSelf.delegate) {
                [weakSelf.delegate onError:weakSelf:nsError];
            }
        }];
    }

    void onMetadata(const std::string& type, const std::vector<uint8_t>& data) override
    {
        TTVPlayer* __weak weakSelf = m_player;
        NSString* typeString = [NSString stringWithUTF8String:type.c_str()];
        NSData* nsData = [[NSData alloc] initWithBytes:data.data() length:data.size()];
        [weakSelf.delegate onMetadata:weakSelf:typeString:nsData];
    }

    void onQualityChanged(const Quality& quality) override
    {
        NSString* name = [NSString stringWithUTF8String:quality.name.c_str()];
        TTVPlayer* __weak weakSelf = m_player;
        [[NSOperationQueue mainQueue] addOperationWithBlock:^(void) {
            if (weakSelf.delegate) {
                [weakSelf.delegate onQualityChanged:weakSelf:name];
            }
        }];
    }

    void onRebuffering() override
    {
    }

    void onRecoverableError(const Error& error) override
    {
        (void)error;
    }

    void onSeekCompleted(MediaTime time) override
    {
        CMTime cmTime = CMTimeMake(time.count(), time.timebase());
        TTVPlayer* __weak weakSelf = m_player;
        [[NSOperationQueue mainQueue] addOperationWithBlock:^(void) {
            if (weakSelf.delegate) {
                [weakSelf.delegate onSeekCompleted:weakSelf:cmTime];
                TTVPlayer* player = weakSelf;
                if (player && player->completionHandler) {
                    player->completionHandler(YES);
                    player->completionHandler = nil;
                }
            }
        }];
    }

    void onSessionData(const std::map<std::string, std::string>& properties) override
    {
        (void)properties; // TODO
    }

    void onStateChanged(Player::State state) override
    {
        (void)state;
        TTVPlayer* __weak weakSelf = m_player;
        [[NSOperationQueue mainQueue] addOperationWithBlock:^(void) {
            if (weakSelf.delegate) {
                [weakSelf.delegate onStateChanged:weakSelf];
            }
        }];
    }

    void onAnalyticsEvent(const std::string& name, const std::string& properties) override
    {
        NSString* nameString = [NSString stringWithUTF8String:name.c_str()];
        NSString* propertiesString = [NSString stringWithUTF8String:properties.c_str()];
        NSError* jsonError;
        NSData* propertiesData = [propertiesString dataUsingEncoding:NSUTF8StringEncoding];
        NSDictionary* dictionary = [NSJSONSerialization JSONObjectWithData:propertiesData
                                                                   options:NSJSONReadingMutableContainers
                                                                     error:&jsonError];

        TTVPlayer* __weak weakSelf = m_player;
        [[NSOperationQueue mainQueue] addOperationWithBlock:^(void) {
            if (weakSelf.delegate) {
                [weakSelf.delegate onAnalyticsEvent:weakSelf:nameString:dictionary];
            }
        }];
    }

    void onPositionChanged(MediaTime position) override
    {
        CMTime current = CMTimeMake(position.count(), position.timebase());
        [m_observers enumerateObjectsUsingBlock:^(id obj, BOOL* stop) {
            id<TTVObserver> observer = (id<TTVObserver>)obj;
            [observer update:current];
            (void)stop;
        }];
    }

    void addObserver(id obj)
    {
        resetObserver(obj);
        [m_observers addObject:obj];
    }

    void removeObserver(id observer)
    {
        [m_observers removeObject:observer];
    }

    void resetObserver(id obj)
    {
        id<TTVObserver> observer = (id<TTVObserver>)obj;
        [observer reset];
    }

    void resetObservers()
    {
        [m_observers enumerateObjectsUsingBlock:^(id obj, BOOL* stop) {
            resetObserver(obj);
            (void)stop;
        }];
    }
};
@end
