#import "TTVURLSessionDelegate.h"
#import "CocoaHttpClient.h"
#import "CocoaHttpRequest.h"
#import "CocoaHttpResponse.h"
#include <map>
#include <mutex>

@implementation TTVURLSessionDelegate {
    std::map<NSUInteger, std::shared_ptr<twitch::CocoaHttpRequest>> _requests;
    std::mutex _mutex;
}

- (void)addRequestWithIdentifier:(NSUInteger)taskIdentifier
                                :(std::shared_ptr<twitch::CocoaHttpRequest>)request
{
    std::lock_guard<std::mutex> lock(_mutex);
    _requests[taskIdentifier] = request;
}

// NSURLSessionTaskDelegate
- (void)URLSession:(NSURLSession*)session
                    task:(NSURLSessionTask*)task
    didCompleteWithError:(nullable NSError*)error
{
    (void)session;
    auto taskIdentifier = [task taskIdentifier];
    std::shared_ptr<twitch::CocoaHttpRequest> request;
    {
        std::lock_guard<std::mutex> lock(_mutex);
        auto it = _requests.find(taskIdentifier);
        if (it == _requests.end()) {
            return;
        }
        request = it->second;
        _requests.erase(taskIdentifier);
    }

    if (request) {
        request->onCompletedWithError(error);
    }
}

// NSURLSessionDataDelegate
- (void)URLSession:(NSURLSession*)session
              dataTask:(NSURLSessionDataTask*)dataTask
    didReceiveResponse:(NSURLResponse*)response
     completionHandler:(void (^)(NSURLSessionResponseDisposition disposition))completionHandler
{
    (void)session;
    auto taskIdentifier = [dataTask taskIdentifier];
    std::shared_ptr<twitch::CocoaHttpRequest> request;

    {
        std::lock_guard<std::mutex> lock(_mutex);
        auto it = _requests.find(taskIdentifier);
        if (it == _requests.end() || !it->second || it->second->isCancelled()) {
            _requests.erase(taskIdentifier);
            NSLog(@"%@ canceled erased completionHandler\n", dataTask);
            completionHandler(NSURLSessionResponseCancel);
            return;
        }
        request = it->second;
    }

    auto statusCode = static_cast<int>([(NSHTTPURLResponse*)response statusCode]);
    auto iosReponse = std::make_shared<twitch::CocoaHttpResponse>((NSHTTPURLResponse*)response, completionHandler, statusCode);
    request->onResponse(iosReponse);
}

- (void)URLSession:(NSURLSession*)session
          dataTask:(NSURLSessionDataTask*)dataTask
    didReceiveData:(NSData*)data
{
    (void)session;
    auto taskIdentifier = [dataTask taskIdentifier];
    std::shared_ptr<twitch::CocoaHttpRequest> request;

    {
        std::lock_guard<std::mutex> lock(_mutex);
        auto it = _requests.find(taskIdentifier);
        if (it == _requests.end() || !it->second) {
            return;
        }
        request = it->second;
        if (request->isCancelled()) {
            [dataTask cancel];
            _requests.erase(taskIdentifier);
            return;
        }
    }

    [data enumerateByteRangesUsingBlock:^(const void* bytes, NSRange byteRange, BOOL* stop) {
        request->onData(static_cast<const uint8_t*>(bytes), byteRange.length, false);
        if (stop) {
            *stop = request->isCancelled() ? YES : NO;
        }
    }];
}
@end
