#include "CurlContext.hpp"
#include "debug/trace.hpp"
#include <cassert>
#include <cstring>

namespace twitch {
CurlContext::CurlContext()
    : m_multi(curl_multi_init())
    , m_canceled(false)
{
}

CurlContext::~CurlContext()
{
    m_canceled = true;
    curl_multi_cleanup(m_multi);
}

CURLcode CurlContext::perform(CurlHttpRequest::State& current, CurlHttpRequest::State next, std::chrono::seconds timeout)
{
    fd_set fdread;
    fd_set fdwrite;
    fd_set fdexcep;

    while (current != next) {
        CURLMcode code = CURLM_CALL_MULTI_PERFORM;
        int handles = 0;

        if (m_canceled) {
            break;
        }

        while (code == CURLM_CALL_MULTI_PERFORM) {
            code = curl_multi_perform(m_multi, &handles);
        }

        if (handles == 0) {
            int msgcount;
            CURLMsg* msg = curl_multi_info_read(m_multi, &msgcount);

            if (msg && (CURLMSG_DONE == msg->msg)) {
                current = CurlHttpRequest::State::READ_RESPONSE_BODY;
                return msg->data.result;
            }

            return CURLE_OK;
        }

        switch (code) {
        case CURLM_OK:
            break;

        case CURLM_CALL_MULTI_PERFORM:
            continue;

        default:
            return CURLE_OK;
        }

        FD_ZERO(&fdread);
        FD_ZERO(&fdwrite);
        FD_ZERO(&fdexcep);
        int maxfd;
        curl_multi_fdset(m_multi, &fdread, &fdwrite, &fdexcep, &maxfd);

        if (maxfd >= 0) {
            timeval timeVal{};
            timeVal.tv_sec = timeout.count();
            int numfds = select(maxfd + 1, &fdread, &fdwrite, &fdexcep, &timeVal);

            if (numfds == 0) {
                return CURLE_OPERATION_TIMEDOUT;
            }
        }
    }

    return CURLE_OK;
}
}
