#include "CurlHttpRequest.hpp"
#include "CurlContext.hpp"
#include "CurlHttpResponse.hpp"
#include "debug/trace.hpp"
#include <cassert>
#include <cstring>

namespace twitch {
CurlHttpRequest::CurlHttpRequest(const std::string& url, HttpMethod method)
    : HttpRequest(url, method)
    , m_easy(nullptr)
    , m_state(State::SEND_REQUEST)
    , m_curlHeaders(nullptr)
    , m_cancelled(false)
    , m_contentPosition(0)
{
    m_easy = curl_easy_init();
}

CurlHttpRequest::~CurlHttpRequest()
{
    if (m_context) {
        curl_multi_remove_handle(m_context->getHandle(), m_easy);
    }
    curl_slist_free_all(m_curlHeaders);
    curl_easy_cleanup(m_easy);
    cancel();
}

void CurlHttpRequest::setHeader(const std::string& key, const std::string& value)
{
    m_headers[key] = value;
}

void CurlHttpRequest::send(const HttpClient::ResponseHandler& onResponse, const HttpClient::ErrorHandler& onError)
{
    // Don't send a canceled request
    if (isCancelled()) {
        return;
    }
    curl_easy_setopt(m_easy, CURLOPT_URL, getUrl().c_str());
    curl_easy_setopt(m_easy, CURLOPT_TIMEOUT, getTimeout().count());
    curl_easy_setopt(m_easy, CURLOPT_WRITEFUNCTION, writeFunction);
    curl_easy_setopt(m_easy, CURLOPT_WRITEDATA, this);
    curl_easy_setopt(m_easy, CURLOPT_PRIVATE, this);
    curl_easy_setopt(m_easy, CURLOPT_CUSTOMREQUEST, getMethodString().c_str());
    curl_easy_setopt(m_easy, CURLOPT_FOLLOWLOCATION, 1L);
    curl_easy_setopt(m_easy, CURLOPT_HEADERFUNCTION, headerFunction);
    curl_easy_setopt(m_easy, CURLOPT_HEADERDATA, this);

    for (const auto& header : m_headers) {
        m_curlHeaders = curl_slist_append(m_curlHeaders, (header.first + ": " + header.second).c_str());
    }

    curl_easy_setopt(m_easy, CURLOPT_HTTPHEADER, m_curlHeaders);

    if (!m_content.empty()) {
        curl_easy_setopt(m_easy, CURLOPT_INFILESIZE, m_content.size());
        curl_easy_setopt(m_easy, CURLOPT_READFUNCTION, readFunction);
        curl_easy_setopt(m_easy, CURLOPT_READDATA, this);
        curl_easy_setopt(m_easy, CURLOPT_UPLOAD, 1L);
    }

    CURLcode code = m_context->perform(m_state, State::HAVE_RESPONSE, getTimeout());

    if (code == CURLcode::CURLE_OK) {
        long status = 0;
        curl_easy_getinfo(m_easy, CURLINFO_RESPONSE_CODE, &status);
        auto response = std::make_shared<CurlHttpResponse>(shared_from_this(), status);
        m_response = response;
        onResponse(response);
    } else {
        if (onError) {
            onError(static_cast<int>(code));
        }
    }
}

void CurlHttpRequest::cancel()
{
    m_cancelled = true;
    std::lock_guard<std::mutex> lock(m_contextMutex);

    auto response = m_response.lock();
    if (response) {
        response->cancel();
    }
}

void CurlHttpRequest::setContent(const std::vector<uint8_t>& content)
{
    m_content = content;
}

void CurlHttpRequest::setContext(std::shared_ptr<CurlContext> context)
{
    std::lock_guard<std::mutex> lock(m_contextMutex);
    m_context = std::move(context);
    CURLMcode code = curl_multi_add_handle(m_context->getHandle(), m_easy);
    assert(code == CURLM_OK);
    if (code != CURLM_OK) {
        m_context = nullptr;
    }
}

CURLcode CurlHttpRequest::resume(std::chrono::seconds timeout)
{
    // resume curl to read the response body
    CURLcode code = curl_easy_pause(m_easy, CURLPAUSE_CONT);

    if (code == CURLE_OK) {
        code = m_context->perform(m_state, CurlHttpRequest::State::READ_RESPONSE_BODY, timeout);
    }

    if (code != CURLE_OK) {
        TRACE_ERROR("curl_easy_pause() failed with error code: %d", static_cast<int>(code));
    }
    return code;
}

size_t CurlHttpRequest::readFunction(char* data, size_t size, size_t nmemb, CurlHttpRequest* request)
{
    if (!request) {
        return 0;
    }
    size_t length = size * nmemb;

    const uint8_t* buffer = &(request->m_content[request->m_contentPosition]);

    size_t bufferSize = request->m_content.size() - request->m_contentPosition;

    if (length >= bufferSize) {
        std::memcpy(data, buffer, bufferSize);
        return bufferSize;
    } else {
        request->m_contentPosition += length;
        std::memcpy(data, buffer, length);
        return length;
    }
}

size_t CurlHttpRequest::writeFunction(char* buffer, size_t size, size_t nmemb, CurlHttpRequest* request)
{
    if (!request) {
        return 0;
    }

    if (request->m_state == State::SEND_REQUEST) {
        // pause here let the api user check the response code
        request->m_state = State::HAVE_RESPONSE;
        return CURL_WRITEFUNC_PAUSE;
    }

    size_t length = size * nmemb;

    if (length) {
        auto response = request->m_response.lock();
        if (response) {
            response->handleContent(reinterpret_cast<uint8_t*>(buffer), length, false);
        }
    }

    return length;
}

size_t CurlHttpRequest::headerFunction(char* buffer, size_t size, size_t nmemb, CurlHttpRequest* request)
{
    (void)request;
    std::string header(buffer, buffer + (size * nmemb) - 1);
    auto position = header.find_first_of(':');
    std::string key = header.substr(0, position);
    std::string value = header.substr(position + 2, header.size());
    auto response = request->m_response.lock();
    if (response) {
        response->setHeader(key, value);
    }
    return size * nmemb;
}
}
