#include "MP4Sink.hpp"
#include "playercore/platform/Scheduler.hpp"
#include <libgen.h>

namespace twitch {
MP4Sink::MP4Sink(MediaSink::Listener& listener, std::shared_ptr<Scheduler> scheduler, bool flushGops, const std::string& file)
    : m_audioID(-1)
    , m_videoID(-1)
    , m_currentTime(0.0)
    , m_listener(listener)
    , m_scheduler(scheduler)
    , m_mp4file(nullptr)
    , m_manifest(nullptr)
    , m_fragOffset(0)
    , m_fragDuration(0.0)
    , m_flushGops(flushGops)
{
    if (!file.empty()) {
        if (file == "-") {
            m_mp4file = std::freopen(nullptr, "wb", stdout);
        } else {
            m_mp4FileName = basename((char*)file.c_str());
            m_manifestName = m_mp4FileName + ".m3u8";
            m_mp4file = std::fopen(file.c_str(), "wb");
            m_manifest = std::fopen((file + ".m3u8").c_str(), "wb");
            if (m_manifest) {
                std::string manifest;
                manifest += "#EXTM3U\n";
                manifest += "#EXT-X-VERSION:7\n";
                manifest += "#EXT-X-PLAYLIST-TYPE:EVENT\n";
                manifest += "#EXT-X-TARGETDURATION:5\n";
                manifest += "#EXT-X-INDEPENDENT-SEGMENTS\n";
                manifest += "#EXT-X-MEDIA-SEQUENCE:0\n";
                std::fwrite(manifest.data(), sizeof(decltype(manifest)::value_type), manifest.size(), m_manifest);
                std::fflush(m_manifest);
            }
        }
    }
}

void MP4Sink::configure(int track, std::shared_ptr<MediaFormat> format)
{
    const auto& mediaType = format->getType();
    std::vector<media::EncryptionInfo> encinfo;
    // encinfo = fmp4drmFactory::clearKeyTestInfo(); // Uncomment this line to test encryption

    if (mediaType == MediaType::Audio_AAC) {
        m_audioID = track;
        m_audioTrack = m_mp4.addTrackAac(std::micro::den, format->getCodecData(MediaFormat::Audio_AAC_ESDS), encinfo);
        flushInit();
    } else if (mediaType == MediaType::Video_AVC) {
        m_videoID = track;
        m_videoTrack = m_mp4.addTrackAvc(std::micro::den, format->getCodecData(MediaFormat::Video_AVC_AVCC), encinfo);
        flushInit();
    }
}

void MP4Sink::enqueue(int track, std::shared_ptr<MediaSampleBuffer> sample)
{
    if (MediaTime(0) >= m_currentTime) {
        m_currentTime = sample->decodeTime;
    }

    if (track == m_audioID) {
        processAudioSample(*sample);
    }

    if (track == m_videoID) {
        if (m_flushGops) {
            // When flushing Gops, once we receive a sync frame, we know
            // we've already processed a full GOP, so we can render it to the buffer.
            const int MaxFragmentDuration = 30000; // milliseconds
            MediaTime duration = m_videoTrack->getDurationTime();
            if (sample->isSyncSample || duration.milliseconds().count() >= MaxFragmentDuration) {
                flushMedia();
                flushManifest();
            }
            processVideoSample(*sample);
        } else {
            if (sample->isSyncSample) {
                flushManifest();
            }

            processVideoSample(*sample);
            flushMedia();
        }
    }
}

void MP4Sink::processVideoSample(const MediaSampleBuffer& sample)
{
    int64_t dts = sample.decodeTime.scaleTo(m_videoTrack->getTimescale()).count();
    int32_t cts = static_cast<int32_t>((sample.presentationTime - sample.decodeTime).scaleTo(m_videoTrack->getTimescale()).count());
    int32_t duration = sample.duration.scaleTo(m_videoTrack->getTimescale()).count();
    uint32_t flags = sample.isSyncSample ? media::Mp4Track::SampleFlagsIntra : media::Mp4Track::SampleFlagsPredicted;
    const uint8_t* data = sample.buffer.data();
    uint32_t size = static_cast<uint32_t>(sample.buffer.size());
    m_videoTrack->addSample(dts, cts, duration, flags, data, size);
}

void MP4Sink::processAudioSample(const MediaSampleBuffer& sample)
{
    int64_t dts = sample.decodeTime.scaleTo(m_audioTrack->getTimescale()).count();
    int32_t cts = 0;
    int32_t duration = sample.duration.scaleTo(m_audioTrack->getTimescale()).count();
    uint32_t flags = media::Mp4Track::SampleFlagsAudio;
    const uint8_t* data = sample.buffer.data();
    uint32_t size = static_cast<uint32_t>(sample.buffer.size());
    m_audioTrack->addSample(dts, cts, duration, flags, data, size);
}

void MP4Sink::endOfStream()
{
    // TODO close manifest!
}

void MP4Sink::flushInit()
{
    if (m_mp4.needInit() && m_audioTrack && m_videoTrack) {
        std::vector<uint8_t> buffer;
        m_mp4.renderMoov(buffer);
        m_fragOffset = buffer.size();

        if (m_mp4file) {
            std::fwrite(buffer.data(), sizeof(decltype(buffer)::value_type), buffer.size(), m_mp4file);
            std::fflush(m_mp4file);
        }

        if (m_manifest) {
            std::string manifest;
            manifest += "#EXT-X-MAP:URI=\"" + m_mp4FileName + "\",BYTERANGE=\"" + std::to_string(buffer.size()) + "@0\"\n";
            std::fwrite(manifest.data(), sizeof(decltype(manifest)::value_type), manifest.size(), m_manifest);
            std::fflush(m_manifest);
        }
    }
}

void MP4Sink::flushMedia()
{
    std::vector<uint8_t> buffer;
    if (!m_mp4.needInit() && m_mp4.samplesPending()) {
        m_fragDuration += m_mp4.fragmentDuration(1000) / 1000.0;
        m_mp4.renderMoof(buffer);
    }

    if (m_mp4file) {
        std::fwrite(buffer.data(), sizeof(decltype(buffer)::value_type), buffer.size(), m_mp4file);
        std::fflush(m_mp4file);
    }
}

void MP4Sink::flushManifest()
{
    size_t offset = 0;
    if (m_mp4file) {
        offset = std::ftell(m_mp4file);
    }

    size_t fragSize = offset - m_fragOffset;
    if (m_manifest && 0 < m_fragDuration && 0 < fragSize) {
        std::string manifest;
        manifest += "#EXTINF:" + std::to_string(m_fragDuration) + ",\n";
        manifest += "#EXT-X-BYTERANGE:" + std::to_string(fragSize) + "@" + std::to_string(m_fragOffset) + "\n";
        manifest += m_mp4FileName + "\n";
        std::fwrite(manifest.data(), sizeof(decltype(manifest)::value_type), manifest.size(), m_manifest);
        std::fflush(m_manifest);
    }

    m_fragOffset = offset;
    m_fragDuration = 0;
}

void MP4Sink::play()
{
    MediaTime interval(std::chrono::milliseconds(250));
    m_outstandingPlay = m_scheduler->schedule([this, interval]() {
        m_currentTime += interval;
        m_listener.onSinkTimeUpdate(m_currentTime);
    },
        interval.microseconds(), true);
}

void MP4Sink::pause()
{
    if (m_outstandingPlay) {
        m_outstandingPlay->cancel();
    }
}

void MP4Sink::reset()
{
    m_mp4 = media::fmp4();
    m_audioTrack.reset();
    m_videoTrack.reset();
    if (m_outstandingPlay) {
        m_outstandingPlay->cancel();
    }
    m_outstandingPlay = m_scheduler->schedule([this]() {
        m_currentTime = MediaTime(0.0);
        m_listener.onSinkTimeUpdate(m_currentTime);
    });
}

void MP4Sink::remove(const TimeRange& range)
{
    (void)range;
}

void MP4Sink::seekTo(MediaTime time)
{
    m_scheduler->schedule([this, time]() {
        m_currentTime = time;
        m_listener.onSinkTimeUpdate(m_currentTime);
    });
}

void MP4Sink::setPlaybackRate(float rate)
{
    (void)rate;
}

void MP4Sink::setVolume(float volume)
{
    (void)volume; // We handle this in the client for now
}

} //namespace twitch
