#include "PosixPlatform.hpp"
#include "CurlHttpClient.hpp"
#include "MP4Sink.hpp"
#include "debug/ExceptionHandler.hpp"
#include "debug/PrintLog.hpp"
#include "sink/NullAudioRenderer.hpp"
#include "sink/NullVideoRenderer.hpp"
#include "sink/PassthroughDecoder.hpp"
#include <pthread.h>
#include <sys/utsname.h>

namespace twitch {

Capabilities posixCapabilities()
{
    Capabilities capabilities;
    capabilities.avcFormat = AVCFormatType::AVCC;
    capabilities.supportsSingleFrameFragments = true;
    return capabilities;
}

PosixPlatform::PosixPlatform(bool flushGops, const std::string& output)
    : m_name("Posix")
    , m_tempPath(P_tmpdir)
    , m_capabilities(posixCapabilities())
    , m_log(std::make_shared<PrintLog>(Log::Level::Warning))
    , m_httpClient(std::make_shared<CurlHttpClient>())
    , m_flushGops(flushGops)
    , m_output(output)
{
    utsname utsname;
    uname(&utsname);
    m_name = utsname.sysname;
}

std::unique_ptr<MediaSink> PosixPlatform::createSink(MediaSink::Listener& listener, std::shared_ptr<Scheduler> scheduler)
{
    if (!m_output.empty()) {
        return std::unique_ptr<MP4Sink>(new MP4Sink(listener, scheduler, m_flushGops, m_output));
    } else {
        return NativePlatform::createSink(listener, scheduler);
    }
}

std::unique_ptr<MediaDecoder> PosixPlatform::createDecoder(std::shared_ptr<const MediaFormat>)
{
    return std::unique_ptr<MediaDecoder>(new PassthroughDecoder());
}

std::unique_ptr<MediaRenderer> PosixPlatform::createRenderer(const ReferenceClock& clock, std::shared_ptr<const MediaFormat> format)
{
    (void)clock;
    const auto& mediaType = format->getType();
    if (mediaType.isAudio()) {
        return std::unique_ptr<MediaRenderer>(new NullAudioRenderer());
    } else if (mediaType.isVideo()) {
        return std::unique_ptr<MediaRenderer>(new NullVideoRenderer());
    }
    return nullptr;
}

void PosixPlatform::onThreadCreated(std::thread::id id, const std::string& name)
{
    ExceptionHandler::registerTerminateHandler(id, name);
}

void PosixPlatform::setCurrentThreadName(const std::string& name)
{
#ifdef __APPLE__
    pthread_setname_np(name.c_str());
#else
    pthread_setname_np(pthread_self(), name.c_str());
#endif
}
}
