#include "AdditionalContent.hpp"
#include "playercore/platform/ps4/PS4Platform.hpp"
#include "debug/trace.hpp"
#include <libsysmodule.h>
#include <cassert>

namespace twitch {
namespace ps4 {

bool AdditionalContent::m_initialized = false;
int AdditionalContent::m_lastError = 0;

bool AdditionalContent::initialize()
{
    if (m_initialized) {
        return true;
    }

    // Reset the last error flag
    m_lastError = 0;

    m_lastError = sceSysmoduleIsLoaded(SCE_SYSMODULE_APP_CONTENT);
    if (m_lastError == SCE_SYSMODULE_ERROR_UNLOADED) {
        m_lastError = sceSysmoduleLoadModule(SCE_SYSMODULE_APP_CONTENT);
        if (m_lastError < 0) {
            PS4Platform::traceError("sceSysmoduleLoadModule", m_lastError);
            return false;
        }
    } else if (m_lastError < 0) {
        PS4Platform::traceError("sceSysmoduleIsLoaded", m_lastError);
        return false;
    }

    SceAppContentInitParam initParam;
    SceAppContentBootParam bootParam;
    memset(&initParam, 0, sizeof(SceAppContentInitParam));
    memset(&bootParam, 0, sizeof(SceAppContentBootParam));

    m_lastError = sceAppContentInitialize(&initParam, &bootParam);
    if (m_lastError == SCE_APP_CONTENT_ERROR_BUSY) {
        TRACE_DEBUG("sceAppContentInitialize returning SCE_APP_CONTENT_ERROR_BUSY - Assuming already is already initialized.");
        m_lastError = 0;
    } else if (m_lastError < 0) {
        PS4Platform::traceError("sceAppContentInitialize failed", m_lastError);
        return false;
    }

    m_initialized = true;
    return true;
}

TempDirMount::TempDirMount(SceAppContentMountPoint&& mountPoint)
    : m_mountPoint(std::move(mountPoint))
    , m_tempPath(m_mountPoint.data)
{
}

TempDirMount::~TempDirMount()
{
    int error = sceAppContentTemporaryDataUnmount(&m_mountPoint);
    if (error != SCE_OK) {
        PS4Platform::traceError("sceAppContentTemporaryDataUnmount failed", error);
        // TODO: add error handling?
    }
}

std::unique_ptr<TempDirMount> TempDirMount::mount()
{
    if (!AdditionalContent::isInitialized()) {
        TRACE_ERROR("Additional Content has not been initialized");
        return std::unique_ptr<TempDirMount>();
    }

    SceAppContentTemporaryDataOption option = SCE_APP_CONTENT_TEMPORARY_DATA_OPTION_NONE;
    SceAppContentMountPoint	mountPoint;
    memset(&mountPoint, 0, sizeof(SceAppContentMountPoint));

    int error = sceAppContentTemporaryDataMount2(option, &mountPoint);
    if (error == SCE_OK) {
        return std::unique_ptr<TempDirMount>(new TempDirMount(std::move(mountPoint)));
    } else if (error == SCE_APP_CONTENT_ERROR_BUSY) {
        TRACE_DEBUG("sceAppContentTemporaryDataMount2 returning SCE_APP_CONTENT_ERROR_BUSY - Assuming temp directory is already mounted.");
        // TODO: Will need to handle this if other libraries also use the temp dir
        assert(false);
    } else if (error < 0) {
        PS4Platform::traceError("sceAppContentTemporaryDataMount2 failed", error);
    }

    return std::unique_ptr<TempDirMount>();
}

}
}
