// Code based on PS4 3.500 SDK Sony Samples

#include "AudioOutputStream.hpp"
#include "util/Concurrency.hpp"
#include "debug/trace.hpp"
#include <thread>

using namespace twitch;
using namespace twitch::ps4;

AudioOutputStream::AudioOutputStream(uint32_t size, uint32_t sizeHead, const std::shared_ptr<AudioClock>& rendererClock)
    : OutputStream(size, sizeHead)
    , m_audioOutput(rendererClock)
    , m_hasOutput(false)
    , m_thrOut(0)
    , m_error(0)
{
    TRACE_INFO("AudioOutputStream::AudioOutputStream(): this:%p, size:%u, sizeHead:%u", this, size, sizeHead);

    int ret = spawnInternalThread();

    if (ret < 0) {
        setError(ret);
        return;
    }

    TRACE_INFO("AudioOutputStream::AudioOutputStream(): this:%p, size:%u, sizeHead:%u ret:%d", this, size, sizeHead, ret);
}

AudioOutputStream::~AudioOutputStream(void)
{
    TRACE_INFO("AudioOutputStream::~AudioOutputStream(): this:%p", this);

    m_sm.changeState(AudioOutputStream::State::Terminate);

    m_buffer->terminate();

    int joinRet = 0;
    if (m_thrOut) {
        joinRet = scePthreadJoin(m_thrOut, 0);
        m_thrOut = 0;
    }
    
    TRACE_INFO("AudioOutputStream::~AudioOutputStream(): this:%p, threadRet:%d", this, joinRet);
}

void AudioOutputStream::flush()
{
    // TODO: have a better implementation of flush() rather than just tearing down the buffer
    //      and creating a new one

    // Terminate the buffer before acquiring the mutex since it might be waiting
    // for the next reading room
    m_buffer->terminate();

    std::lock_guard<Mutex> lock(m_mutex);
    auto size = m_buffer->size();
    auto marginSize = m_buffer->marginSize();
    m_buffer.reset(new RingBuffer(size, marginSize));

    m_audioOutput.flush();
}

void AudioOutputStream::start()
{
    assert(m_sm.getState() != AudioOutputStream::State::Terminate);

    if (m_sm.getState() == AudioOutputStream::State::Terminate) {
        return;
    }

    m_sm.changeState(AudioOutputStream::State::Run);
}

void AudioOutputStream::stop()
{
    assert(m_sm.getState() != AudioOutputStream::State::Terminate);

    if (m_sm.getState() == AudioOutputStream::State::Terminate) {
        return;
    }

    m_audioOutput.resetLastRenderedSampleTime();
    m_sm.changeState(State::Pause);
}

int AudioOutputStream::output(uint32_t size)
{
    std::lock_guard<Mutex> lock(m_mutex);

    int ret = 0;

    size = size ? size : m_audioOutput.outputSamplesSize();

    const uint32_t requiredRead = m_audioOutput.outputSamplesSize();
    if (size != requiredRead) {
        return -1;
    }

    const void* p = m_buffer->lockReadingRoom(requiredRead);

    if (p) {
        //TRACE_DEBUG("Reading %d bytes from RingBuffer (Original MediaTime: %d", size, sampleInsertionMediaTime.count());
        ret = m_audioOutput.output(p);

        if (ret < 0) {
            TRACE_ERROR("AudioOutput::output() failed: 0x%08X", ret);
            m_buffer->unlockReadingRoom(0);
        } else if (m_hasOutput) {
            m_buffer->unlockReadingRoom(requiredRead, requiredRead);
        } else {
            m_hasOutput = true;
        }
    }

    return ret;
}

void* AudioOutputStream::threadAudioOutBgm(void* arg)
{
    int ret = 0;

    AudioOutputStream* outputStream = reinterpret_cast<AudioOutputStream*>(arg);
    assert(outputStream);

    ret = outputStream->mainOut();

    if (ret < 0) {
        TRACE_ERROR("Audio::mainOut() failed: 0x%08X", ret);
    }

    return reinterpret_cast<void*>(ret);
}

int AudioOutputStream::mainOut()
{
    int ret = 0;
    while (m_sm.getState() != State::Terminate) {
        if (isEmpty()) {
            continue;
        }

        // Double-check to avoid locking too much between samples
        if (m_sm.getState() == State::Pause) {
            m_audioOutput.resetLastRenderedSampleTime();
            m_sm.waitUntilNotInState(State::Pause);
            continue;
        }

        ret = output();
        if (ret < 0) {
            TRACE_ERROR("AudioOutputStream::output() failed: 0x%08X", ret);
            break;
        }
    }

    if (m_sm.getState() == State::Terminate) {
        TRACE_DEBUG("AudioOutputStream::mainOut request termination");
    }
    return ret;
}

int AudioOutputStream::spawnInternalThread()
{
    ScePthreadAttr attr = 0;
    SceKernelSchedParam schedparam;
    const uint32_t stacksize = 512 * 1024;

    // create a thread attribute object
    scePthreadAttrInit(&attr);
    // set stack size
    scePthreadAttrSetstacksize(&attr, stacksize);
    // set schedule priority
    scePthreadAttrSetinheritsched(&attr, SCE_PTHREAD_EXPLICIT_SCHED);
    schedparam.sched_priority = SCE_KERNEL_PRIO_FIFO_HIGHEST;
    scePthreadAttrSetschedparam(&attr, &schedparam);

    // create a thread for audio output
    int ret = scePthreadCreate(&m_thrOut, &attr, threadAudioOutBgm, this, "Player AudioOutBgm");

    if (ret < 0) {
        TRACE_ERROR("scePthreadCreate() failed: 0x%08X", ret);
    }

    ret = scePthreadAttrDestroy(&attr);
    if (ret < 0) {
        TRACE_ERROR("scePthreadAttrDestroy() failed: 0x%08X", ret);
    }

    return ret;
}

void AudioOutputStream::setError(int error)
{
    m_error = error;
}

void AudioOutputStream::StateMachine::changeState(State newState)
{
    {
        std::lock_guard<Mutex> lock(m_condMutex);

        if (newState == m_state) {
            return;
        }

        m_state = newState;
    }
    m_conditionVariable.notify_one();
}

void AudioOutputStream::StateMachine::waitUntilNotInState(State state)
{
    std::unique_lock<Mutex> lock(m_condMutex);
    if (m_state == state) {
        m_conditionVariable.wait(lock);
    }
}

bool AudioOutputStream::isEmpty()
{
    std::lock_guard<Mutex> lock(m_mutex);
    return m_buffer->isEmpty();
}
