#pragma once

#include <atomic>
#include "Stream.hpp"
#include "util/Concurrency.hpp"

namespace twitch {
namespace ps4 {
class AudioClock;
class AudioOutputStream : public OutputStream {
public:
    enum class State
    {
        Run,            // Normal state, thread continue processing happily
        Terminate,      // Request the thread to stop processing. Will end up killing the thread.
        Pause,          // Request the thread to pause. Thread is still alive but will be idle.
    };

    // State machine class, hides mutex and condition variable behind a single class, can be generalized
    class StateMachine
    {
    public:
        StateMachine() :
            m_state(State::Run) {}

        void changeState(State newState);
        State getState() const { return m_state; }
        void waitUntilNotInState(State state);
    private:
        volatile State m_state;
        ConditionVariable m_conditionVariable;
        Mutex m_condMutex;
    };


    AudioOutputStream(uint32_t size, uint32_t sizeHead, const std::shared_ptr<AudioClock>& rendererClock);
    ~AudioOutputStream(void);
    int open(const char *path, const char *mode) override
    {
        (void)path;
        (void)mode;
        return -1;
    }
    int open(int32_t grain, int32_t sampleRate, int32_t param, float volume) override { return m_audioOutput.open(grain, sampleRate, param, volume); }
    int close(void) override
    {
        m_hasOutput = false;
        return m_audioOutput.close();
    }
    int output(uint32_t size = 0) override;
    void flush();
    MediaTime getLastRenderedSampleTime() const { return m_audioOutput.getLastRenderedSampleTime(); }
    uint32_t outputSamplesSize(void) const { return m_audioOutput.outputSamplesSize(); }

    int setVolume(float volume) { return m_audioOutput.setVolume(volume); }

    bool hasError() const { return m_error != 0; }
    int getError() const { return m_error; }

    // restart processing of the data in the AudioOutputStream
    void start() override;

    // Stop processing data in the AudioOutputStream. Queued data will be "stalled" until 'start' is called again
    void stop() override;

    bool isEmpty();
private:
    AudioOutputStream(const AudioOutputStream& rhs) = delete;
    AudioOutputStream &operator=(const AudioOutputStream& rhs) = delete;

    int spawnInternalThread();
    void setError(int error);

    // AudioOutputStream is self-managed - You give it data, it outputs to the speaker
    static void *threadAudioOutBgm(void *arg);
    int mainOut();

    AudioOutput m_audioOutput;
    bool m_hasOutput;
    ScePthread m_thrOut;
    int m_error;
    StateMachine m_sm;
    Mutex m_mutex;
};

}
}

