#include "AudioRenderer.hpp"
#include "AudioClock.hpp"
#include "AudioResampler.hpp"
#include "playercore/platform/ps4/PS4Platform.hpp"
#include "debug/trace.hpp"
#include <cassert>

using namespace twitch;

twitch::ps4::AudioRenderer::AudioRenderer()
    : m_clock(std::make_shared<AudioClock>())
    , m_inputNumChannels(0)
    , m_inputSampleRate(0)
    , m_inputSampleSize(0)
    , m_outputNumChannels(0)
    , m_volume(1.f)
{
}

MediaResult ps4::AudioRenderer::configure(const MediaFormat& format)
{
    int numChannels = format.getInt(MediaFormat::Audio_ChannelCount);
    int sampleRate = format.getInt(MediaFormat::Audio_SampleRate);
    int sampleSize = format.getInt(MediaFormat::Audio_SampleSize);

    TRACE_DEBUG("AudioRenderer::configure called with ChannelCount=%d, SampleRate=%d and SampleSize=%d", numChannels, sampleRate, sampleSize);

    // Has the sampleRate and number of channels stayed the same ? If so, no need to rebuild the audio output stream.
    if (numChannels == m_inputNumChannels && sampleRate == m_inputSampleRate && sampleSize == m_inputSampleSize) {
        TRACE_DEBUG("AudioRenderer::configure doesn't need to be reconfigured");
        return MediaResult::Ok;
    }

    // we only support 1 or 2 channels
    if (numChannels != 1 && numChannels != 2 && numChannels != 6 && numChannels != 8) {
        TRACE_DEBUG("AudioRenderer only support mono (1), stereo (2), 5.1 (6) or 7.1 (8) audio output. NumChannels: %d", numChannels);
        return MediaResult::ErrorInvalidParameter;
    }

    // only support 48khz sample rate
    if (sampleRate != SupportedOutputSampleRate) {
        TRACE_ERROR("Audio sample rate (%d khz) is not supported", m_inputSampleRate);
        assert(false);
        return MediaResult::ErrorInvalidParameter;
    }

    m_inputNumChannels = numChannels;
    m_inputSampleRate = sampleRate;
    m_inputSampleSize = sampleSize;

    if (m_inputNumChannels == 6) {
        m_outputNumChannels = 8;
    } else {
        m_outputNumChannels = m_inputNumChannels;
    }

    // Sony consider L & R channel as '1' sample.
    // We will be outputting 8192 bytes of data. For Stereo, 32 bits sounds, that means 8 bytes per "sample"
    // Use the lowest possible grain size so that the audio clock increments as granularly as possible
    const uint32_t outputStreamGrain = 256;
    const uint32_t outputStreamSize = outputStreamGrain * sampleSize * m_outputNumChannels * sizeof(float);

    TRACE_DEBUG("audio output: %u sample, %u Hz, %u channel(s). Ring buffer size: %u", outputStreamGrain, SupportedOutputSampleRate, m_outputNumChannels, outputStreamSize);

    // We don't want two active AudioOutputStream at the same time, delete the previous instance (if any)
    m_output = nullptr;
    m_output.reset(new AudioOutputStream(outputStreamSize, sizeof(float) * (m_outputNumChannels)*outputStreamGrain, m_clock));

    if (m_output->hasError()) {
        PS4Platform::traceError("AudioOutputStream creation failed", m_output->getError());
        return MediaResult(MediaResult::ErrorInvalidParameter, m_output->getError());
    }

    // open an audio output
    int ret = m_output->open(outputStreamGrain, SupportedOutputSampleRate, AudioSystem::param(m_outputNumChannels), m_volume);

    if (ret < 0) {
        PS4Platform::traceError("OutputStream::open() failed", ret);
        return MediaResult(MediaResult::ErrorInvalidParameter, ret);
    }

    TRACE_DEBUG("AudioRenderer::configure completed returning MediaResult::Ok");

    return MediaResult::Ok;
}

MediaResult ps4::AudioRenderer::getRenderedPresentationTime(MediaTime& time)
{
    time = m_output->getLastRenderedSampleTime();
    return MediaResult::Ok;
}

MediaResult ps4::AudioRenderer::render(std::shared_ptr<const MediaSample> sample)
{
    auto input = std::static_pointer_cast<const MediaSampleBuffer>(sample);
    if (input->buffer.size() == 0) {
        return MediaResult::Ok;
    }

    // Check if the renderer clock needs to start
    if (!m_clock->isInitialized()) {
        // Provide the PTS to the audio clock as an initial offset
        m_clock->initialize(input->presentationTime, SupportedOutputSampleRate);
    }

    uint8_t* dest = m_output->lock(input->buffer.size());
    size_t writeSize = m_output->writingSize();
    if (input->buffer.size() > writeSize) {
        TRACE_ERROR("Audio sample buffer size (%ld) exceeds output writing size (%ld)", input->buffer.size(), writeSize);
        assert(false);
        return MediaResult::ErrorInvalidData;
    }

    memcpy(dest, input->buffer.data(), input->buffer.size());
    m_output->unlock(input->buffer.size());

    return MediaResult::Ok;
}

MediaResult ps4::AudioRenderer::flush()
{
    assert(m_outputNumChannels == 1 || m_outputNumChannels == 2 || m_outputNumChannels == 6 || m_outputNumChannels == 8);
    m_output->flush();
    return MediaResult::Ok;
}

MediaResult ps4::AudioRenderer::start()
{
    m_output->start();
    return MediaResult::Ok;
}

MediaResult ps4::AudioRenderer::stop()
{
    m_output->stop();
    return MediaResult::Ok;
}

MediaResult ps4::AudioRenderer::setPlaybackRate(float rate)
{
    return (rate != 1.0f) ? MediaResult::ErrorNotSupported : MediaResult::Ok;
}

MediaResult ps4::AudioRenderer::setVolume(float volume)
{
    m_volume = volume;

    // early out if Configure hasn't been called yet
    if (!m_output) {
        return MediaResult::Ok;
    }

    int ret = m_output->setVolume(volume);
    if (ret < 0) {
        TRACE_WARN("AudioRenderer::setVolume failed with ret=%08x", ret);
        return MediaResult(MediaResult::ErrorInvalidState, ret);
    }

    return MediaResult::Ok;
}
