#include "AudioResampler.hpp"
#include "debug/trace.hpp"
#include <cassert>

namespace twitch {
namespace ps4 {

std::unique_ptr<AudioResampler> AudioResampler::create(uint32_t numChannels, uint32_t inputSampleRate, uint32_t outputSampleRate)
{
    if (inputSampleRate == outputSampleRate) {
        TRACE_ERROR("Cannot create audio resampler for equal input and output sample rates (%d khz)", inputSampleRate);
        assert(false);
        return std::unique_ptr<AudioResampler>();
    }

    switch (numChannels) {
    case 1:
        return std::unique_ptr<AudioResampler>(new LinearResampler<MonoSample, MonoSample>(outputSampleRate, inputSampleRate));
    case 2:
        return std::unique_ptr<AudioResampler>(new LinearResampler<StereoSample, StereoSample>(outputSampleRate, inputSampleRate));
    case 6:
        return std::unique_ptr<AudioResampler>(new LinearResampler<SevenDotOneSample, FiveDotOneSample>(outputSampleRate, inputSampleRate));
    case 8:
        return std::unique_ptr<AudioResampler>(new LinearResampler<SevenDotOneSample, SevenDotOneSample>(outputSampleRate, inputSampleRate));
    default:
        TRACE_ERROR("Cannot create audio resampler for %d number of channels", numChannels);
        assert(false);
        return std::unique_ptr<AudioResampler>();
    }
}

template <typename OutputSample, typename InputSample>
float LinearResampler<OutputSample, InputSample>::lerp(float a, float b, float t)
{
    return (1 - t) * a + t * b;
}

template<>
void LinearResampler<SevenDotOneSample, FiveDotOneSample>::lerp(SevenDotOneSample& output, const FiveDotOneSample& a, const FiveDotOneSample& b, float t)
{
    output.l = lerp(a.l, b.l, t);
    output.r = lerp(a.r, b.r, t);
    output.c = lerp(a.c, b.c, t);
    output.lfe = lerp(a.lfe, b.lfe, t);
    output.lextend = 0.0f;
    output.rextend = 0.0f;
    output.lsurround = lerp(a.lsurround, b.lsurround, t);
    output.rsurround = lerp(a.rsurround, b.rsurround, t);
}

template<>
void LinearResampler<SevenDotOneSample, SevenDotOneSample>::lerp(SevenDotOneSample& output, const SevenDotOneSample& a, const SevenDotOneSample& b, float t)
{
    output.l = lerp(a.l, b.l, t);
    output.r = lerp(a.r, b.r, t);
    output.c = lerp(a.c, b.c, t);
    output.lfe = lerp(a.lfe, b.lfe, t);
    output.lextend = lerp(a.lextend, b.lextend, t);
    output.rextend = lerp(a.rextend, b.rextend, t);
    output.lsurround = lerp(a.lsurround, b.lsurround, t);
    output.rsurround = lerp(a.rsurround, b.rsurround, t);
}

template<>
void LinearResampler<StereoSample, StereoSample>::lerp(StereoSample& output, const StereoSample& a, const StereoSample& b, float t)
{
    output.left = lerp(a.left, b.left, t);
    output.right = lerp(a.right, b.right, t);
}

template<>
void LinearResampler<MonoSample, MonoSample>::lerp(MonoSample& output, const MonoSample& a, const MonoSample& b, float t)
{
    output.mono = lerp(a.mono, b.mono, t);
}

}
}
