#pragma once

#include "AudioOutputStream.hpp"
#include "AudioRenderer.hpp"
#include "AudioSample.hpp"
#include "playercore/MediaSample.hpp"
#include <cassert>
#include <vector>

namespace twitch {
namespace ps4 {
    class AudioResampler {
    public:
        virtual ~AudioResampler() = default;

        using BufferType = std::vector<uint8_t>;

        static std::unique_ptr<AudioResampler> create(uint32_t numChannels, uint32_t inputSampleRate, uint32_t outputSampleRate);
        virtual void flush() = 0;
        virtual uint32_t getInputSampleRate() = 0;
        virtual void resample(BufferType& resampledBuffer, const void * buffer, size_t bufferSize) = 0;
    };

    template <typename OutputSample, typename InputSample>
    class LinearResampler : public AudioResampler {
    public:
        using InputSampleBuffer = std::vector<InputSample>;
        using OutputSampleBuffer = std::vector<OutputSample>;

        LinearResampler(uint32_t outputSampleRate, uint32_t inputSampleRate)
            : m_outputSampleRate(outputSampleRate)
            , m_inputSampleRate(inputSampleRate)
        {
            assert(m_outputSampleRate > 0);
            assert(m_inputSampleRate > 0);
            assert(m_inputSampleRate != m_outputSampleRate);
        }

        void flush() override
        {
            m_remainingInputFromLastPass.clear();
            m_lastResamplingIndex = 0.f;
        }

        uint32_t getInputSampleRate() override { return m_inputSampleRate; }

        static float lerp(float a, float b, float t);
        static void lerp(OutputSample& output, const InputSample& a, const InputSample& b, float t);

        void resample(BufferType& resampledBuffer, const void * buffer, size_t bufferSize) override
        {
            // Append the input sample buffer to the end of the remaining buffer from the last pass
            InputSampleBuffer complexInput = m_remainingInputFromLastPass;
            m_remainingInputFromLastPass.clear();
            AudioSample<InputSample>::appendToBuffer(complexInput, buffer, bufferSize);

            // Resample
            OutputSampleBuffer outputBuffer;
            resample(outputBuffer, complexInput);

            // TODO: Avoid buffer copy and resample to output buffer directly
            const auto& resampleBufferSize = outputBuffer.size() * sizeof(OutputSample);
            resampledBuffer.resize(resampleBufferSize);
            std::memcpy(resampledBuffer.data(), outputBuffer.data(), resampleBufferSize);
        }

        void resample(OutputSampleBuffer& outputBuffer, const InputSampleBuffer& inputBuffer)
        {
            // Example: Going from 44100 hz to 48000 hz means going from 1024 samples to 1114.5578231292517 samples
            // Logically, this means some output should be 1114 samples (4456 bytes) while others should be 1115 samples (4460 bytes) for proper result.
            // We keep partial results into the m_remainingInputFromLastPass structure and we also keep the last modular index into m_lastResamplingIndex

            float ratio = static_cast<float>(m_inputSampleRate) / static_cast<float>(m_outputSampleRate);

            // How many *complete* samples will we output ? It is easier to compute them to be safe
            int outputNumSamples = 0;
            for (float i = m_lastResamplingIndex; i + 1 < inputBuffer.size(); i += ratio) {
                outputNumSamples++;
            }

            outputBuffer.resize(outputNumSamples);

            // a simple LERP will do
            for (int i = 0; i < outputNumSamples; ++i, m_lastResamplingIndex += ratio) {
                size_t sIndex = std::floor(m_lastResamplingIndex);
                lerp(outputBuffer[i], inputBuffer[sIndex], inputBuffer[sIndex + 1], fmodf(m_lastResamplingIndex, 1.f));
                assert(sIndex + 1 < inputBuffer.size());
            }

            // store remaining last sample(s)
            assert(m_remainingInputFromLastPass.empty());
            size_t sIndex = std::floor(m_lastResamplingIndex);
            m_remainingInputFromLastPass.insert(m_remainingInputFromLastPass.begin(), inputBuffer.begin() + sIndex, inputBuffer.end());

            // and last step we were at
            m_lastResamplingIndex = fmodf(m_lastResamplingIndex, 1.f);
        }

    protected:
        const uint32_t m_outputSampleRate;
        const uint32_t m_inputSampleRate;
        float m_lastResamplingIndex = 0.f;
        InputSampleBuffer m_remainingInputFromLastPass;
    };
}
}
