// Code based on PS4 3.500 SDK Sony Samples

#include "AudioClock.hpp"
#include "AudioSystem.hpp"
#include "playercore/platform/ps4/PS4Platform.hpp"
#include "debug/trace.hpp"
#include <audioout.h>
#include <cassert>
#include <iostream>

using namespace twitch;
using namespace ps4;

AudioSystem::AudioSystem()
    : m_lastError(0)
{
    init();
}

AudioSystem::~AudioSystem()
{
}

void AudioSystem::init()
{
    static bool _initialized = false;

    if (!_initialized) {
        int ret = sceAudioOutInit();

        if (ret < 0 && ret != SCE_AUDIO_OUT_ERROR_ALREADY_INIT) {
            PS4Platform::traceError("sceAudioOutInit failed", ret);
            setError(ret);
            return;
        }

        _initialized = true;
    }
}

void AudioSystem::setError(int error)
{
    m_lastError = error;
}

int32_t AudioSystem::param(uint32_t numChannels)
{
    int32_t param = 0;

    switch (numChannels) {
    case 1:
        param = SCE_AUDIO_OUT_PARAM_FORMAT_FLOAT_MONO;
        break;

    case 2:
        param = SCE_AUDIO_OUT_PARAM_FORMAT_FLOAT_STEREO;
        break;

    case 8:
        param = SCE_AUDIO_OUT_PARAM_FORMAT_FLOAT_8CH;
        break;

    default:
        TRACE_WARN("unsupported number of channels: %d", param);
        break;
    }

    return param;
}

uint32_t AudioSystem::numChannels(int32_t param)
{
    uint32_t numChannels = 0;

    switch (param) {
    case SCE_AUDIO_OUT_PARAM_FORMAT_FLOAT_MONO:
        numChannels = 1;
        break;

    case SCE_AUDIO_OUT_PARAM_FORMAT_FLOAT_STEREO:
        numChannels = 2;
        break;

    case SCE_AUDIO_OUT_PARAM_FORMAT_FLOAT_8CH:
        numChannels = 8;
        break;

    default:
        TRACE_ERROR("unsupported number of channels: %u\n", numChannels);
        break;
    }

    return numChannels;
}

AudioOutput::AudioOutput(const std::shared_ptr<AudioClock>& rendererClock)
    : m_rendererClock(rendererClock)
    , m_handle(-1)
    , m_grain(0)
    , m_sampleRate(0)
    , m_param(0)
    , m_numChannels(0)
{
}

AudioOutput::~AudioOutput(void)
{
    close();
}

MediaTime AudioOutput::getLastRenderedSampleTime() const
{
    if (m_handle <= 0) {
        return MediaTime::zero();
    }

    // Will return MediaTime::zero() if !rendererClock->isInitialized()
    return m_rendererClock->elapsedPts();
}

int AudioOutput::open(int32_t grain, int32_t sampleRate, int32_t param, float volume)
{
    if (m_handle > 0) {
        close();
    }

    // open audio port
    // index: device index (0)
    // grain: number of samples to be output at once; 64 to 64*1023
    // sampleRate: must be 48000
    // param: data format
    int ret = sceAudioOutOpen(SCE_USER_SERVICE_USER_ID_SYSTEM, SCE_AUDIO_OUT_PORT_TYPE_MAIN, 0, grain, sampleRate, param);

    if (ret == SCE_AUDIO_OUT_ERROR_NOT_INIT) {
        s_system.init();
        ret = sceAudioOutOpen(SCE_USER_SERVICE_USER_ID_SYSTEM, SCE_AUDIO_OUT_PORT_TYPE_MAIN, 0, grain, sampleRate, param);
    }

    if (ret < 0) {
        PS4Platform::traceError("sceAudioOutOpen() failed", ret);
        return ret;
    }

    m_handle = ret;
    m_grain = grain;
    m_sampleRate = sampleRate;
    m_param = param;
    m_numChannels = AudioSystem::numChannels(param);

    // set volume
    ret = setVolume(volume);

    if (ret < 0) {
        PS4Platform::traceError("sceAudioOutSetVolume() failed", ret);
        return ret;
    }

    return ret;
}

int AudioOutput::close(void)
{
    int ret = 0;
    int tmp;

    if (0 < m_handle) {
        // output remaining audio
        sceAudioOutOutput(m_handle, nullptr);
        // close audio port
        tmp = sceAudioOutClose(m_handle);

        if (tmp < 0) {
            PS4Platform::traceError("sceAudioOutClose() failed", tmp);
            ret = (ret < 0) ? ret : tmp;
        }
    }

    // set initial values
    m_handle = 0;
    m_grain = 0;
    m_sampleRate = 0;
    m_param = 0;
    m_numChannels = 0;

    return ret;
}

int AudioOutput::flush()
{
    // output remaining audio
    TRACE_DEBUG("AudioOutput::flush(): sceAudioOutOutput");
    int error = sceAudioOutOutput(m_handle, nullptr);
    if (error < 0) {
        TRACE_ERROR("AudioOutput::flush(): sceAudioOutOutput error: %d", error);
    }
    resetLastRenderedSampleTime();
    return error;
}

int AudioOutput::output(const void* p)
{
    // output audio data
    int ret = sceAudioOutOutput(m_handle, p);

    // When this function return, any previous output has already been played and the registered output is ready to play. The buffer must be kept until it can be discarted.
    if (ret < 0) {
        PS4Platform::traceError("sceAudioOutOutput() failed", ret);
        return ret;
    }

    m_rendererClock->update(static_cast<size_t>(ret));
    return ret;
}

void AudioOutput::resetLastRenderedSampleTime()
{
    m_rendererClock->reset();
}

int AudioOutput::setVolume(float volume)
{
    // 0.0f --> 0
    // 1.0f --> SCE_AUDIO_OUT_VOLUME_0DB
    int32_t sceVolume = static_cast<int32_t>(volume * SCE_AUDIO_OUT_VOLUME_0DB);

    // fail safe in case we are being passed invalid volume value
    sceVolume = std::min(std::max(sceVolume, 0), SCE_AUDIO_OUT_VOLUME_0DB);

    int32_t sceVolumes[8] = { sceVolume, sceVolume, sceVolume, sceVolume, sceVolume, sceVolume, sceVolume, sceVolume };

    // https://ps4.scedev.net/resources/documents/SDK/4.500/AudioOut-Reference/0018.html
    int ret = sceAudioOutSetVolume(m_handle, (SCE_AUDIO_VOLUME_FLAG_L_CH | SCE_AUDIO_VOLUME_FLAG_R_CH | SCE_AUDIO_VOLUME_FLAG_C_CH | SCE_AUDIO_VOLUME_FLAG_LFE_CH | SCE_AUDIO_VOLUME_FLAG_LS_CH | SCE_AUDIO_VOLUME_FLAG_RS_CH | SCE_AUDIO_VOLUME_FLAG_LE_CH | SCE_AUDIO_VOLUME_FLAG_RE_CH), sceVolumes);
    return ret;
}
