﻿#include <assert.h>
#include <kernel.h>
#include <memory.h>
#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/dmem.h>
#include <sys/mman.h>

#include "FrameBufferPool.hpp"
#include "playercore/platform/ps4/PS4Platform.hpp"
#include "debug/trace.hpp"

using namespace twitch;
using namespace twitch::ps4;

static unsigned int roundUp(unsigned int size, unsigned int align)
{
    return ((size + (align - 1u)) & (~(align - 1u)));
}

FrameBufferPool::FrameBufferPool(int width, int height, int numBuffers)
    : m_alloc_ptr(nullptr)
    , m_total_size(0)
    , m_alloc_offset(0)
{
    assert(width > 0);
    assert(height > 0);
    assert(numBuffers >= 1);

    m_nodes.resize(numBuffers);

    uint32_t alignedWidth;
    uint32_t frameBufferSize;
    uint32_t alignedBufferSize;

    alignedWidth = roundUp(width, 256);

    frameBufferSize = alignedWidth * height * 2;
    frameBufferSize += frameBufferSize / 2;
    frameBufferSize += 4096;

    // allocate frame buffer
    alignedBufferSize = roundUp(frameBufferSize,
        SCE_KERNEL_PAGE_SIZE);

    m_total_size = m_nodes.size() * alignedBufferSize;

    TRACE_DEBUG("Creating FrameBufferPool width:%d alignedWidth:%d alignment:%d alignedBufferSize:%d m_total_size:%d",
        width, alignedWidth, SCE_KERNEL_PAGE_SIZE, alignedBufferSize, m_total_size);

    m_lastError = sceKernelAllocateDirectMemory(0, SCE_KERNEL_MAIN_DMEM_SIZE,
        m_total_size, SCE_KERNEL_PAGE_SIZE,
        SCE_KERNEL_WC_GARLIC, &m_alloc_offset);

    if (m_lastError) {
        TRACE_ERROR("sceKernelAllocateDirectMemory:0x%08x failed\n", m_lastError);
        return;
    }

    TRACE_DEBUG("sceKernelAllocateDirectMemory done ... size:%d offset:%ld", m_total_size, m_alloc_offset);

    m_lastError = sceKernelMapDirectMemory(&m_alloc_ptr,
        m_total_size,
        SCE_KERNEL_PROT_CPU_READ | SCE_KERNEL_PROT_CPU_WRITE | SCE_KERNEL_PROT_GPU_ALL,
        0, m_alloc_offset, SCE_KERNEL_PAGE_SIZE);

    if (m_lastError) {
        TRACE_ERROR("sceKernelMapDirectMemory:0x%08x failed\n", m_lastError);
        return;
    }

    if (m_alloc_ptr == nullptr) {
        TRACE_ERROR("frame buffer allocation failed.\n");
        return;
    }

    // allocate aligned frame buffer.
    for (size_t i = 0; i < m_nodes.size(); ++i) {
        m_nodes[i].pFrame = (uint8_t*)m_alloc_ptr + (alignedBufferSize * i);
        m_nodes[i].frameSize = alignedBufferSize;
        m_free.push(&m_nodes[i]);
    }
}

FrameBufferPool::~FrameBufferPool()
{
    stop();

    // Wait for all of the frames to return before deallocating the memory
    waitForEmptyBuffer();

    int32_t retVal = 0;

    if (m_alloc_ptr != nullptr) {
        retVal = munmap(m_alloc_ptr, m_total_size);

        if (retVal == 0) {
            retVal = sceKernelReleaseDirectMemory(m_alloc_offset, m_total_size);
            if (retVal != 0) {
                PS4Platform::traceError("sceKernelReleaseDirectMemory failed", retVal);
                assert(false);
            }
        } else {
            PS4Platform::traceError("sceKernelReleaseDirectMemory failed", retVal);
            assert(false);
        }
    }
}

std::shared_ptr<MediaFrameBufferInfo> FrameBufferPool::getFrameBuffer()
{
    auto onFrameBufferDone = [this](MediaFrameBufferInfo* bufferInfo) {
        releaseFrameBuffer(bufferInfo);
    };

    std::unique_lock<Mutex> lock(m_mutex);
    if (m_stopped) {
        return std::shared_ptr<MediaFrameBufferInfo>();
    } else if (m_free.empty()) {
        // TODO: Pass timeout as parameter or make configurable
        std::chrono::milliseconds timeout(100);
        if (m_condition.wait_for(lock, timeout) != CvStatus::no_timeout || m_stopped) {
            return std::shared_ptr<MediaFrameBufferInfo>();
        }
    }

    auto bufferInfo = m_free.front();
    m_free.pop();
    m_condition.notify_one();
    return std::shared_ptr<MediaFrameBufferInfo>(bufferInfo, onFrameBufferDone);
}

void FrameBufferPool::releaseFrameBuffer(MediaFrameBufferInfo* bufferInfo)
{
    {
        std::lock_guard<Mutex> lock(m_mutex);
        m_free.push(bufferInfo);
    }
    m_condition.notify_all();
}

void FrameBufferPool::stop()
{
    std::lock_guard<Mutex> lock(m_mutex);
    if (m_stopped) {
        TRACE_DEBUG("FrameBufferPool already stopped");
    } else {
        m_stopped = true;
        m_condition.notify_all();
    }
}

void FrameBufferPool::waitForEmptyBuffer()
{
    std::unique_lock<Mutex> lock(m_mutex);
    while (m_free.size() < m_nodes.size()) {
        m_condition.wait(lock);
    }
}
