#include "Http.hpp"
#include "playercore/platform/ps4/PS4Platform.hpp"
#include "debug/trace.hpp"
#include "debug/tracecall.hpp"
#include <algorithm>
#include <cassert>
#include <libhttp.h>
#include <net.h>
#include <string.h>

using namespace twitch::ps4;

HttpClient::HttpClient(PS4Platform& platform, const std::string& deviceId)
    : ScopedScheduler(platform.createScheduler("HttpResponseScheduler"))
    , m_deviceId(deviceId)
    , m_context(0)
    , m_templateSetting(0)
    , m_error(0)
{
    int ret = m_net.init();

    if (ret < 0) {
        m_error = ret;
        return;
    }

    int context = sceHttpInit(m_net.getPoolId(), m_net.getSslContext(), MemoryPoolSize);

    if (context < 0) {
        m_error = context;
        return;
    }

    // TODO: Revisit to make UA dynamic and standardized.
    int templateSetting = sceHttpCreateTemplate(context, "Twitch UserAgent (PS4)", SCE_HTTP_VERSION_1_1, SCE_HTTP_TRUE);

    if (templateSetting < 0) {
        m_error = templateSetting;
        cleanContext();
        return;
    }

    int cookiesInitialized = initCookies(context, templateSetting);
    if (cookiesInitialized != SCE_OK) {
        m_error = cookiesInitialized;
        return;
    }

    m_templateSetting = templateSetting;
    m_context = context;
}

HttpClient::~HttpClient()
{
    cancel();
    cleanTemplate();
    cleanContext();

    m_net.deinit();
}

void HttpClient::cleanContext()
{
    if (m_context) {
        sceHttpTerm(m_context);
        m_context = 0;
    }
}

void HttpClient::cleanTemplate()
{
    if (m_templateSetting) {
        sceHttpDeleteTemplate(m_templateSetting);
        m_templateSetting = 0;
    }
}

int HttpClient::initCookies(int contextId, int templateId)
{
    if (m_deviceId.empty()) {
        TRACE_INFO("No device ID found");
        return SCE_OK;
    }

    int error = sceHttpSetCookieEnabled(templateId, SCE_TRUE);
    if (error != SCE_OK) {
        TRACE_ERROR("sceHttpSetCookieEnabled(): Failed with error: %d", error);
        return error;
    }

    std::string cookie = "unique_id=" + m_deviceId;
    error = sceHttpAddCookie(contextId, "https://api.twitch.tv", cookie.c_str(), cookie.length());
    if (error != SCE_OK) {
        TRACE_ERROR("sceHttpSetCookieEnabled(): Failed with error: %d", error);
        return error;
    }

    return SCE_OK;
}

std::shared_ptr<twitch::HttpRequest> HttpClient::createRequest(const std::string& url, twitch::HttpMethod method)
{
    if (m_error) {
        TRACE_ERROR("HttpClient::createRequest Request will be unusable because HttpClient is in error state: %08x", m_error);
        // can't return nullptr here, code expect a valid pointer and we would make it crash
    }

    return std::make_shared<ps4::HttpRequest>(shared_from_this(), url, method, m_templateSetting);
}

void HttpClient::send(std::shared_ptr<twitch::HttpRequest> request, ResponseHandler onResponse, ErrorHandler onError)
{
    schedule([this, request, onResponse, onError]() {
        auto ps4Request = std::static_pointer_cast<twitch::ps4::HttpRequest>(request);

        // Check if the HTTP client received an error upon initialization
        if (m_error != 0) {
            if (!ps4Request->isCancelled()) {
                onError(m_error);
                return;
            }
        }

        TRACE_DEBUG("Requesting URL: %s", request->getUrl().c_str());
        std::shared_ptr<ps4::HttpResponse> response;
        if (ps4Request->send()) {
            response = ps4Request->createResponse();

            if (!ps4Request->isCancelled()) {
                if (ps4Request->hasError()) {
                    onError(ps4Request->getError());
                } else {
                    onResponse(response);
                }
            }
        } else { // you'll go here if the request has been canceled
            if (ps4Request->hasError() && !ps4Request->isCancelled()) {
                onError(ps4Request->getError());
            }
        }

    });
}

HttpRequest::HttpRequest(std::shared_ptr<HttpClient> httpClient, const std::string& url, twitch::HttpMethod method, int templateId)
    : twitch::HttpRequest(url, method)
    , m_httpClient(httpClient)
{
    // TODO: Create HttpConnection class
    int connection = sceHttpCreateConnectionWithURL(templateId, m_url.c_str(), true);
    if (connection < 0) {
        TRACE_ERROR("HttpRequest(): Failed creating connection with error: %d", connection);
        m_error = connection;
        return;
    } else {
        m_connectionId = connection;
    }

    assert(method == twitch::HttpMethod::GET); // only GET has been implemented
    int request = sceHttpCreateRequestWithURL2(m_connectionId, "GET", m_url.c_str(), 0);
    if (request < 0) {
        TRACE_ERROR("HttpRequest(): Failed creating request with error: %d", request);
        m_error = request;
    } else {
        m_requestId = request;
    }
}

HttpRequest::~HttpRequest()
{
    if (m_requestId > 0) {
        sceHttpDeleteRequest(m_requestId);
    }

    if (m_connectionId > 0) {
        sceHttpDeleteConnection(m_connectionId);
    }
}

std::shared_ptr<twitch::ps4::HttpResponse> HttpRequest::createResponse()
{
    if (isCancelled() || hasError()) {
        return std::shared_ptr<HttpResponse>();
    }

    int statusCode;
    m_error = sceHttpGetStatusCode(m_requestId, &statusCode);
    if (hasError()) {
        PS4Platform::traceError("sceHttpGetStatusCode", m_error);
        return std::shared_ptr<HttpResponse>();
    }

    std::map<std::string, std::string> headers;
    if (!getResponseHeaders(headers)) {
        return std::shared_ptr<HttpResponse>();
    }

    return std::make_shared<HttpResponse>(statusCode, std::move(headers), shared_from_this());
}

bool HttpRequest::getResponseHeaders(std::map<std::string, std::string>& headers)
{
    if (hasError()) {
        return false;
    }

    char* header;
    size_t headerSize;
    m_error = sceHttpGetAllResponseHeaders(m_requestId, &header, &headerSize);
    if (hasError()) {
        return false;
    }

    std::vector<char> headerCopy(header, header + headerSize);

    // Find all headers and add them
    rsize_t strmax = headerSize;
    const char* delim = "\n";
    char* next_token;
    char* token = strtok_s(headerCopy.data(), &strmax, delim, &next_token);
    while (token) {
        // token -> Content-Type: <type>
        char* separator = strchr(token, ':');
        if (separator) {
            std::string fieldName(token, separator);
            std::string fieldValue;
            if (parseResponseHeader(fieldValue, header, headerSize, fieldName)) {
                headers[fieldName] = fieldValue;
            } else if (m_error != 0) {
                TRACE_WARN("setHeader on %s failed with 0x%08x", fieldName.c_str(), m_error);
            }
        }

        token = strtok_s(nullptr, &strmax, delim, &next_token);
    }

    // Content-Length is special
    int result;
    uint64_t contentLength;
    m_error = sceHttpGetResponseContentLength(m_requestId, &result, &contentLength);
    if (m_error < 0) {
        TRACE_WARN("sceHttpGetResponseContentLength failed with 0x%08x", m_error);
        return false;
    } else if (m_error == 0 && result == SCE_HTTP_CONTENTLEN_EXIST) {
        headers["Content-Length"] = std::to_string(contentLength);
    }

    return true;
}

bool HttpRequest::parseResponseHeader(std::string& value, char* header, size_t headerSize, const std::string& fieldName)
{
    const char* fieldValue;
    size_t fieldSize;

    int error = sceHttpParseResponseHeader(header, headerSize, fieldName.c_str(), &fieldValue, &fieldSize);
    if (error < 0) {
        if (error != SCE_HTTP_ERROR_PARSE_HTTP_NOT_FOUND) {
            m_error = error;
        }
        return false;
    }

    value = std::string(fieldValue, fieldSize);
    return true;
}

void HttpRequest::scheduleOnError(HttpResponse::ErrorHandler& onError, int errorValue)
{
    m_cancelError.cancel();
    m_cancelError = m_httpClient->schedule([onError, this, errorValue]() {
        if (!isCancelled()) {
            onError(errorValue);
        }
    });
}

void HttpRequest::readResponseData(HttpResponse::ContentHandler onBuffer, HttpResponse::ErrorHandler onError)
{
    m_cancelRead.cancel();
    m_cancelRead = m_httpClient->schedule([this, onBuffer, onError]() {
        TRACE_DEBUG("HttpScheduler running on thread %zu", std::this_thread::get_id().hash());
        assert(!hasError());
        static const int HttpBufferSize = 16384;
        char buf[HttpBufferSize];

        while (true) {
            {
                if (isCancelled()) {
                    TRACE_DEBUG("HTTP Request cancelled");
                    return;
                }
            }

            int read = sceHttpReadData(m_requestId, buf, sizeof(buf));

            if (read < 0) {
                if (read == SCE_HTTP_ERROR_ABORTED || read == SCE_NET_ERROR_EINTR) {
                    TRACE_DEBUG("HTTP Request cancelled");
                } else if (read == SCE_HTTP_ERROR_BEFORE_INIT) {
                    // Allow HTTP response to return gracefully if the HTTP client has destructed,
                    // but the user is still trying to read an outstanding HTTP response
                    TRACE_DEBUG("HTTP client has been destroyed");
                } else {
                    m_error = read;
                    PS4Platform::traceError("sceHttpReadData", m_error);
                    if (!isCancelled()) {
                        onError(m_error);
                    }
                }
                return;
            } else if (read == 0) {
                if (!isCancelled()) {
                    onBuffer(nullptr, 0, true);
                }
                return;
            } else if (read > 0) {
                if (!isCancelled()) {
                    onBuffer(reinterpret_cast<uint8_t*>(buf), static_cast<size_t>(read), false);
                }
            }
        }
    });
}

bool HttpRequest::send()
{
    if (hasError()) {
        return false;
    }

    auto timeout = std::chrono::duration_cast<std::chrono::microseconds>(getTimeout());
    {
        if (isCancelled() || !setTimeout(TimeoutType::Connect, timeout)) {
            return false;
        }
    }

    for (const auto& it : m_headers) {

        std::string key = it.first;
        std::transform(key.begin(), key.end(), key.begin(), ::tolower);
        const std::string& value = it.second;

        int error;
        if (key == "content-length") {
            error = sceHttpSetRequestContentLength(m_requestId, std::stoi(value));
            if (error < 0) {
                m_error = error;
                TRACE_ERROR("HttpRequest::send(): Failed at sceHttpSetRequestContentLength with error: %d", m_error);
                return false;
            }
        } else {
            error = sceHttpAddRequestHeader(m_requestId, key.c_str(), value.c_str(), SCE_HTTP_HEADER_OVERWRITE);
            if (error < 0) {
                m_error = error;
                TRACE_ERROR("HttpRequest::send(): Failed at sceHttpAddRequestHeader with error: %d", m_error);
                return false;
            }
        }
    }

    int error = sceHttpSendRequest(m_requestId, NULL, 0);
    if (error < 0) {
        if (error != SCE_HTTP_ERROR_ABORTED) {
            m_error = error;
            TRACE_ERROR("HttpRequest::send(): Failed with error: %d", m_error);
        }
        return false;
    } else {
        return true;
    }
}

bool HttpRequest::setTimeout(TimeoutType type, std::chrono::microseconds timeout)
{
    if (hasError()) {
        return false;
    }

    std::string typeStr;

    switch (type) {
    case TimeoutType::Connect:
        m_error = sceHttpSetConnectTimeOut(m_requestId, static_cast<uint32_t>(timeout.count()));
        typeStr = "Connect";
        break;
    case TimeoutType::Receive:
        m_error = sceHttpSetRecvTimeOut(m_requestId, static_cast<uint32_t>(timeout.count()));
        typeStr = "Receive";
        break;
    case TimeoutType::Send:
    default:
        m_error = sceHttpSetSendTimeOut(m_requestId, static_cast<uint32_t>(timeout.count()));
        typeStr = "Send";
        break;
    }

    if (m_error != SCE_OK) {
        TRACE_ERROR("HttpRequest::setTimeout(): Failed setting type=%s with error: %d", typeStr.c_str(), m_error);
        return false;
    } else {
        return true;
    }
}

void HttpRequest::cancel()
{
    if (isCancelled()) {
        return;
    }

    m_cancelled = true;
    int error = sceHttpAbortRequest(m_requestId);
    if (error != 0) {
        PS4Platform::traceError("sceHttpAbortRequest", error);
    }

    m_cancelRead.cancel();
    m_cancelError.cancel();
}

HttpResponse::HttpResponse(int statusCode, const std::map<std::string, std::string>& headers, std::shared_ptr<HttpRequest> request)
    : twitch::HttpResponse(statusCode)
    , m_request(request)
{
    m_headers.insert(headers.begin(), headers.end());
}

HttpResponse::~HttpResponse()
{
}

void HttpResponse::read(ContentHandler onBuffer, ErrorHandler onError)
{
    auto timeout = std::chrono::duration_cast<std::chrono::microseconds>(getReadTimeout());
    if (m_request->hasError() || !m_request->setTimeout(HttpRequest::TimeoutType::Receive, timeout)) {
        m_request->scheduleOnError(onError, m_request->getError());
    } else {
        m_request->readResponseData(onBuffer, onError);
    }
}
