#pragma once

#include "playercore/platform/HttpClient.hpp"
#include "player/ScopedScheduler.hpp"
#include "platforms/ps4/Net.hpp"
#include <thread>
#include "player/cancellation.hpp"

namespace twitch {
class PS4Platform;
namespace ps4 {
    class HttpRequest;
    class HttpResponse;

    class HttpClient : public twitch::HttpClient, public std::enable_shared_from_this<HttpClient>, private ScopedScheduler {
    public:
        HttpClient(PS4Platform& platform, const std::string& deviceId);
        virtual ~HttpClient();

        std::shared_ptr<twitch::HttpRequest> createRequest(const std::string& url, twitch::HttpMethod method /* = HttpMethod::GET */) override;
        void send(std::shared_ptr<twitch::HttpRequest> request, ResponseHandler onResponse, ErrorHandler onError) override;
        bool isInitialized() const { return m_context > 0; }

    private:
        void cleanContext();
        void cleanTemplate();
        int initCookies(int contextId, int templateId);

        friend class HttpRequest;

        Net m_net;

        std::string m_deviceId;
        int m_context;
        int m_templateSetting;
        int m_error; // some initialization is done in the constructor, if they fail, this variable is set to the last error.

        static const int MemoryPoolSize = 1024 * 1024;
    };

    class HttpRequest : public twitch::HttpRequest, public std::enable_shared_from_this<HttpRequest> {
    public:
        enum class TimeoutType {
            Connect,
            Receive,
            // Resolve,
            Send
        };

        HttpRequest(std::shared_ptr<HttpClient> httpClient, const std::string& url, twitch::HttpMethod method, int templateId);
        ~HttpRequest() override;

        void cancel() override;
        bool send();

        int getError() const { return m_error; }
        bool hasError() const { return m_error != 0; }

        void readResponseData(twitch::HttpResponse::ContentHandler onBuffer, twitch::HttpResponse::ErrorHandler onError);
        bool setTimeout(TimeoutType type, std::chrono::microseconds timeout);
        void setHeader(const std::string& key, const std::string& value) override
        {
            m_headers[key] = value;
        }
        void setContent(const std::vector<uint8_t>& content) override
        {
            (void) content;
            // TODO support request content
        }
    private:
        std::shared_ptr<twitch::ps4::HttpResponse> createResponse();
        bool isCancelled() { return m_cancelled; }

        bool getResponseHeaders(std::map<std::string, std::string>& headers);
        bool parseResponseHeader(std::string& value, char* header, size_t headerSize, const std::string& fieldName);

        void scheduleOnError(twitch::HttpResponse::ErrorHandler& onError, int value);
        friend class HttpClient;
        friend class HttpResponse;
        std::shared_ptr<HttpClient> m_httpClient;
        twitch::HttpHeaderMap m_headers;
        bool m_cancelled = false;
        int m_connectionId = -1;
        int m_requestId = -1;
        int m_error = 0;

        CancellableRef m_cancelRead;
        CancellableRef m_cancelError;
    };

    class HttpResponse : public twitch::HttpResponse {
    public:
        HttpResponse(int statusCode, const std::map<std::string, std::string>& headers, std::shared_ptr<HttpRequest> request);
        ~HttpResponse();
        void read(ContentHandler onBuffer, ErrorHandler onError) override;

        std::string getHeader(const std::string& key) const override
        {
            auto iter = m_headers.find(key);
            return (iter == m_headers.end()) ? "" : iter->second;
        }

    private:
        std::shared_ptr<HttpRequest> m_request;
        twitch::HttpHeaderMap m_headers;
    };
}
}
