#include "playercore/platform/ps4/PS4Platform.hpp"
#include "AudioDecoder.hpp"
#include "AudioRenderer.hpp"
#include "Http.hpp"
#include "playercore/platform/ps4/PlayerCore.hpp"
#include "sink/threadscheduler.hpp"
#include "debug/trace.hpp"
#include "VideoDecoder.hpp"
#include "VideoRenderer.hpp"
#include "debug/printlog.hpp"
#include <cassert>
#include <sstream>

#include "PS4Errors.inl"

using namespace twitch;
using namespace twitch::ps4;

std::shared_ptr<twitch::Log> PS4Platform::m_log = std::make_shared<PrintLog>();
const std::string PS4Platform::m_name = "ps4";

PS4Platform::PS4Platform(OnVideoFrameBuffer onVideoFrameBuffer, OnFrameNeedsReleasing onFrameNeedsReleasing, const std::string& deviceId)
    : m_onVideoFrameBuffer(onVideoFrameBuffer)
    , m_onFrameNeedsReleasing(onFrameNeedsReleasing)
    , m_capabilities()
    , m_videoDecoder(new VideoDecoder(*PlayerCore::m_configuration.get()))
    , m_httpClient(std::make_shared<ps4::HttpClient>(*this, deviceId))
{
    assert(PlayerCore::isInitialized());

    m_capabilities.supportsLowLatencyABR = true;
}

PS4Platform::~PS4Platform()
{
    // NOTE: Need to provide destructor implementation for std::unique_ptr<TempDirMount> to compile
}

VideoDecoderCapabilities PS4Platform::getVideoDecoderCapabilities(const MediaType& mediaType)
{
    VideoDecoderCapabilities capabilities;
    if (mediaType.matches(MediaType::Video_AVC)) {
        capabilities.maxWidth = PlayerCore::m_configuration->maxWidth;
        capabilities.maxHeight = PlayerCore::m_configuration->maxHeight;
        capabilities.maxLevel = PlayerCore::m_configuration->maxAvcLevel;
        capabilities.maxProfile = PlayerCore::m_configuration->maxAvcProfile;
    }
    return capabilities;
}

const std::string& PS4Platform::getTempPath() const
{
    return PlayerCore::getTempPath();
}

void PS4Platform::traceError(const std::string& errMsg, int errorCode)
{
    std::string canonical;
    std::string description;

    PS4Platform::sceErrorToString(errorCode, canonical, description);

    TRACE_ERROR("%s (%s, %s)", errMsg.c_str(), canonical.c_str(), description.c_str());
}

std::unique_ptr<MediaDecoder> PS4Platform::createDecoder(std::shared_ptr<const MediaFormat> format)
{
    if (format->getType().matches(MediaType::Audio_AAC)) {
        return std::unique_ptr<MediaDecoder>(new AudioDecoder());
    } else if (format->getType().matches(MediaType::Video_AVC)) {
        
        // Usually, only one VideoDecoder is created per platform. We create it in the constructor since it uses a lot of memory and thus it gets the memory as soon as possible
        // which reduces fragmentation
        if (!m_videoDecoder) {
            auto videoDecoder = new VideoDecoder(*PlayerCore::m_configuration.get());
            return std::unique_ptr<MediaDecoder>(videoDecoder);
        } else {
            // transfer ownership
            return std::move(m_videoDecoder);
        }
    }

    return nullptr;
}

std::unique_ptr<MediaRenderer> PS4Platform::createRenderer(const ReferenceClock& clock, std::shared_ptr<const MediaFormat> format)
{
    if (format->getType().matches(MediaType::Audio_AAC)) {
        return std::unique_ptr<MediaRenderer>(new twitch::ps4::AudioRenderer());
    } else if (format->getType().matches(MediaType::Video_AVC)) {
        return std::unique_ptr<MediaRenderer>(new ps4::VideoRenderer(m_onVideoFrameBuffer, m_onFrameNeedsReleasing, clock, PlayerCore::m_configuration->frameBufferCount));
    }

    return nullptr;
}

std::shared_ptr<twitch::Scheduler> twitch::PS4Platform::createScheduler(const std::string& name)
{
    return std::make_shared<ThreadScheduler>(*this, name);
}

void PS4Platform::onThreadCreated(std::thread::id, const std::string&)
{
}

void PS4Platform::setCurrentThreadName(const std::string& name)
{
    scePthreadRename(scePthreadSelf(), name.c_str());
}
