#include "AdditionalContent.hpp"
#include "debug/debugplayer.hpp"
#include "playercore/platform/ps4/PS4Platform.hpp"
#include "debug/trace.hpp"
#include <cassert>
#include <audiodec.h>
#include <libsysmodule.h>

namespace twitch {
namespace ps4 {

std::unique_ptr<twitch::ps4::Configuration> PlayerCore::m_configuration = nullptr;
std::unique_ptr<ps4::TempDirMount> PlayerCore::m_tempDirMount = nullptr;

namespace
{
    struct Module
    {
        uint16_t id;
        const char * name;
    };

    const Module modules[] =
    {
        { SCE_SYSMODULE_VIDEODEC2, "VideoDec2" },
        { SCE_SYSMODULE_AUDIODEC, "AudioDec" },
    };
}

struct ResourceLoader
{
    ResourceLoader()
    {
        for (auto & module : modules)
        {
            int ret = sceSysmoduleLoadModule(module.id);
            if (ret < 0) {
                TRACE_ERROR("Could not load %s 0x%08x", module.name, ret);
            }
        }

        int ret = sceAudiodecInitLibrary(SCE_AUDIODEC_TYPE_M4AAC);
        if (ret < 0) {
            TRACE_ERROR("Could not initialize M4AAC 0x%08x", ret);
        }
    }

    ~ResourceLoader()
    {
        sceAudiodecTermLibrary(SCE_AUDIODEC_TYPE_M4AAC);

        for (auto & module : modules)
        {
            if (sceSysmoduleIsLoaded(module.id)) {
                sceSysmoduleUnloadModule(module.id);
            }
        }
    }
};

bool PlayerCore::initialize(Configuration configuration)
{
    if (isInitialized()) {
        TRACE_ERROR("Player Core has already been initialized");
        return false;
    }

    static ResourceLoader resourceLoader;

    if (configuration.InitializeAdditionalContent && !AdditionalContent::initialize()) {
        TRACE_ERROR("Could not initialize additional app content");
        return false;
    }

    m_tempDirMount = TempDirMount::mount();
    if (!m_tempDirMount) {
        TRACE_WARN("Could not mount temporary directory");
        if (configuration.downloadDir.empty()) {
            TRACE_ERROR("Configuration downloadDir cannot be empty for applications where temp directory cannot be mounted");
            return false;
        }
    }

    m_configuration = std::unique_ptr<Configuration>(new Configuration(configuration));
    return true;
}

bool PlayerCore::isInitialized()
{
    return m_configuration != nullptr;
}

std::shared_ptr<twitch::Player> PlayerCore::create(Player::Listener& listener, PlayerCore::OnVideoFrameBuffer onVideoFrameBuffer, PlayerCore::OnFrameNeedsReleasing onFrameNeedsReleasing,
                                                   MediaSource::Request::Listener* requestListener)
{
    if (!isInitialized()) {
        TRACE_ERROR("Player Core has not been initialized");
        return nullptr;
    }

    auto platform = std::make_shared<PS4Platform>(onVideoFrameBuffer, onFrameNeedsReleasing);
    platform->getLog()->setLevel(m_configuration->logLevel);

#ifdef NDEBUG
    auto player = std::make_shared<AsyncMediaPlayer>(listener, platform, requestListener);
#else
    auto player = std::make_shared<DebugMediaPlayer>(listener, platform, requestListener);
#endif
    
    return player;
}

const std::string& PlayerCore::getTempPath()
{
    assert(isInitialized());

    if (m_tempDirMount) {
        return m_tempDirMount->getTempPath();
    }

    return m_configuration->downloadDir;
}

}
}
