#pragma once

#include "trace/trace.hpp"
#include "assert.h"
#include "util/Concurrency.hpp"
#include <ajm.h>
#include <atomic>

namespace twitch {

static Mutex m_mutex;

template <typename T>
class ResourceLoader {
public:
    ResourceLoader(const std::string& name)
        : m_name(name)
        , m_lastError(0)
    {
    }

    int init()
    {
        std::lock_guard<Mutex> lock(m_mutex);

        m_lastError = 0;
        m_sRefCount++;
        TRACE_DEBUG("ResourceLoader for %s init. Refcount is now %d", m_name.c_str(), m_sRefCount.load());

        if (m_sRefCount == 1) {
            m_lastError = onAcquire();
            if (m_lastError == 0) {
                TRACE_DEBUG("ResourceLoader for %s init success", m_name.c_str());
            } else if (m_lastError != SCE_AJM_ERROR_CODEC_ALREADY_REGISTERED) {
                TRACE_ERROR("ResourceLoader for %s failed with code=0x%08x", m_name.c_str(), m_lastError); // this should never occur with our refcount, assuming no other modules is initializing
            } else {
                TRACE_DEBUG("ResourceLoader for %s init returned code=0x%08x, assuming success", m_name.c_str(), m_lastError);
            }
        }

        TRACE_DEBUG("ResourceLoader init m_lastError is 0x%08x", m_lastError);
        return m_lastError;
    }

    int deinit()
    {
        std::lock_guard<Mutex> lock(m_mutex);
        assert(m_sRefCount >= 1);
        m_sRefCount--;

        TRACE_DEBUG("ResourceLoader for %s deinit. Refcount is now %d", m_name.c_str(), m_sRefCount.load());

        if (m_sRefCount == 0) {
            if (m_lastError != 0) {
                TRACE_DEBUG("ResourceLoader for %s deinit. Not deinitializing since init failed", m_name.c_str());
                return m_lastError;
            }

            m_lastError = onRelease();
            if (m_lastError < 0) {
                TRACE_ERROR("ResourceLoader for %s deinit failed with code 0x%08x", m_name.c_str(), m_lastError);
            } else {
                TRACE_DEBUG("ResourceLoader for %s deinit success", m_name.c_str());
            }
        }

        return m_lastError;
    }

    int getLastError() const { return m_lastError; }

protected:
    virtual int onAcquire() const = 0;
    virtual int onRelease() const = 0;

private:
    std::string m_name;
    int m_lastError;

    static std::atomic<int> m_sRefCount;
};
}