#include "VideoDecoder.hpp"
#include "playercore/platform/ps4/PS4Platform.hpp"
#include "debug/trace.hpp"
#include "debug/tracecall.hpp"
#include <assert.h>
#include <chrono>
#include <libsysmodule.h>
#include <thread>
#include <videodec2.h>

using namespace twitch;
using namespace twitch::ps4;

static unsigned int roundUp(unsigned int size, unsigned int align)
{
    return ((size + (align - 1u)) & (~(align - 1u)));
}

VideoDecoder::Configuration::Configuration(int width, int height, int avcLevel, int avcProfile)
    : width(width)
    , height(height)
    , avcLevel(avcLevel)
    , avcProfile(avcProfile)
{
}

std::unique_ptr<VideoDecoder::Configuration> VideoDecoder::Configuration::create(const MediaFormat& format)
{
    if (!format.hasInt(MediaFormat::Video_Width)) {
        TRACE_ERROR("MediaFormat missing MediaFormat::Video_Width");
        return std::unique_ptr<VideoDecoder::Configuration>();
    }

    if (!format.hasInt(MediaFormat::Video_Height)) {
        TRACE_ERROR("MediaFormat missing MediaFormat::Video_Height");
        return std::unique_ptr<VideoDecoder::Configuration>();
    }

    if (!format.hasInt(MediaFormat::Video_AVC_Level)) {
        TRACE_ERROR("MediaFormat missing MediaFormat::Video_AVC_Level");
        return std::unique_ptr<VideoDecoder::Configuration>();
    }

    if (!format.hasInt(MediaFormat::Video_AVC_Profile)) {
        TRACE_ERROR("MediaFormat missing MediaFormat::Video_AVC_Profile");
        return std::unique_ptr<VideoDecoder::Configuration>();
    }

    int width = format.getInt(MediaFormat::Video_Width);
    int height = format.getInt(MediaFormat::Video_Height);
    int avcLevel = format.getInt(MediaFormat::Video_AVC_Level);
    int avcProfile = format.getInt(MediaFormat::Video_AVC_Profile);

    auto config = new VideoDecoder::Configuration(width, height, avcLevel, avcProfile);
    return std::unique_ptr<VideoDecoder::Configuration>(config);
}

bool VideoDecoder::Configuration::operator==(const Configuration& other) const
{
    return width == other.width && height == other.height && avcLevel == other.avcLevel && avcProfile == other.avcProfile;
}

bool VideoDecoder::Configuration::isValid(ps4::Configuration & validator) const
{
    if (width < 0 || width > validator.maxWidth) {
        TRACE_ERROR("Video_Width=%d is not supported. Max is %d", validator.maxWidth);
        return false;
    }

    if (height < 0 || height > validator.maxHeight) {
        TRACE_ERROR("Video_Height=%d is not supported. Max is %d", validator.maxHeight);
        return false;
    }

    if (avcLevel < 0 || avcLevel > validator.maxAvcLevel) {
        TRACE_ERROR("Video_AVC_Level=%d is not supported. Max is %d", validator.maxAvcLevel);
        return false;
    }

    if (avcProfile < 0 || avcProfile > validator.maxAvcProfile) {
        TRACE_ERROR("Video_AVC_Profile=%d is not supported. Max is %d", validator.maxAvcProfile);
        return false;
    }

    return true;
}

VideoDecoder::VideoDecoder(const ps4::Configuration & globalConfiguration)
    : m_globalConfiguration(globalConfiguration)
    , m_decoderHandle(nullptr)
    , m_frameBufferPool(globalConfiguration.maxWidth, globalConfiguration.maxHeight, globalConfiguration.frameBufferCount)
{
    assert(m_frameBufferPool.getLastError() == 0);
    memset(&m_configInfo, 0, sizeof(m_configInfo));

    m_ctorResult = configureDecoder();

    if (m_ctorResult < 0) {
        TRACE_ERROR("configureDecoder: %x", m_ctorResult);
        return;
    }

    m_ctorResult = MediaResult(spawnInternalThread());

    if (m_ctorResult < 0) {
        TRACE_ERROR("spawnInternalThread: %x", m_ctorResult);
        return;
    }
}

VideoDecoder::~VideoDecoder()
{
    TraceCall trace("ps4::VideoDecoder::~VideoDecoder", 0.f);
    reset();
    m_frameBufferPool.stop();
    destroyDecoder();
    m_computeQueue = nullptr;

    releaseInternalThread();
}

MediaResult VideoDecoder::configureDecoder()
{
    assert(!m_configuration);

    if (m_frameBufferPool.getLastError()) {
        return MediaResult(MediaResult::ErrorInvalidState, m_frameBufferPool.getLastError());
    }

    // TODO: Fix CVP-1678 in order to properly reset decoder on quality switch
    // NOTE: Allocate the maximum video frame configuration settings to bypass decoder errors
    auto configuration = std::unique_ptr<Configuration>(new Configuration(m_globalConfiguration.maxWidth, 
                                                                          m_globalConfiguration.maxHeight, 
                                                                          m_globalConfiguration.maxAvcLevel,
                                                                          m_globalConfiguration.maxAvcProfile));
    if (!configuration || !configuration->isValid(m_globalConfiguration)) {
        return MediaResult::ErrorInvalidParameter;
    }

    m_configuration.swap(configuration);

    MediaResult result;

    result = init();
    if (result == MediaResult::Ok) {
        result = startDecoding();
    }

    return result;
}

MediaResult VideoDecoder::configure(const MediaFormat&, MediaFormat& output)
{
    TRACE_INFO("ps4::VideoDecoder::configure");

    if (m_ctorResult != MediaResult::Ok) {
        return m_ctorResult;
    }

    if (!m_decoderHandle) {
        return MediaResult::ErrorInvalidState;
    }

    output.setInt(MediaFormat::Video_Width, m_configuration->width);
    output.setInt(MediaFormat::Video_Height, m_configuration->height);
    output.setInt(MediaFormat::Video_AVC_Level, m_configuration->avcLevel);
    output.setInt(MediaFormat::Video_AVC_Profile, m_configuration->avcProfile);

    return MediaResult::Ok;
}


void* VideoDecoder::threadVideoDecoder(void* arg)
{
    VideoDecoderContext* videoDecoderContext = reinterpret_cast<VideoDecoderContext*>(arg);
    assert(videoDecoderContext);

    while (videoDecoderContext->running)
    {
        std::unique_lock<Mutex> lock(videoDecoderContext->conditionMutex);
        videoDecoderContext->conditionVariable.wait(lock);

        if (!videoDecoderContext->running) {
            break;
        }

        videoDecoderContext->ret = sceVideodec2Decode(videoDecoderContext->decoderInstanceIn,
                                                      videoDecoderContext->pInputDataInOut,
                                                      videoDecoderContext->pFrameBufferInOut,
                                                      videoDecoderContext->pOutputInfoOut);
        videoDecoderContext->done = true;

        lock.unlock();

        videoDecoderContext->conditionVariable.notify_one();
    }

    return nullptr;
}

MediaResult VideoDecoder::decode(const MediaSampleBuffer& input)
{
    auto inputSample = std::make_shared<InputSample>();
    inputSample->id = ++m_inputCount;
    // TODO: Avoid buffer copy
    inputSample->mediaSample = input;

    SceVideodec2InputData inputData;

    inputData.thisSize = sizeof(inputData);
    inputData.pAuData = const_cast<uint8_t*>(inputSample->mediaSample.buffer.data());
    inputData.auSize = input.buffer.size();

    inputData.dtsData = static_cast<uint64_t>(inputSample->mediaSample.decodeTime.nanoseconds().count());
    inputData.ptsData = static_cast<uint64_t>(inputSample->mediaSample.presentationTime.nanoseconds().count());

    auto mediaFrameBufferInfo = m_frameBufferPool.getFrameBuffer();
    if (!mediaFrameBufferInfo) {
        // This will happen when the VideoDecoder is trying to destruct or
        // FrameBufferPool times out
        TRACE_WARN("ps4::VideoDecoder::decode(): Dropping frame at PTS=%f due to full frame buffer", input.presentationTime.seconds());
        return MediaResult::Ok;
    }

    inputData.attachedData = inputSample->id;
    m_decoderBuffer[inputSample->id] = inputSample;

    // the frame buffer for output
    SceVideodec2FrameBuffer frameBuffer;
    frameBuffer.thisSize = sizeof(frameBuffer);
    frameBuffer.pFrameBuffer = mediaFrameBufferInfo->pFrame;
    frameBuffer.frameBufferSize = mediaFrameBufferInfo->frameSize;

    SceVideodec2OutputInfo outputInfo;
    outputInfo.thisSize = sizeof(outputInfo);

    int retVal = 0;

    do {
        if (retVal == SCE_VIDEODEC2_ERROR_NEW_SEQUENCE) {
            TRACE_DEBUG("new sequence is detected!");

            // please note this reset call discards all the remaining frames.
            // call sceVideodec2Flush to display all the frames if it is needed.
            retVal = resetDecoder();

            TRACE_DEBUG("sceVideodec2Reset done ... decode new sequence again.");
        }

        m_videoDecoderContext.decoderInstanceIn = m_decoderHandle;
        m_videoDecoderContext.pInputDataInOut = &inputData,
        m_videoDecoderContext.pFrameBufferInOut = &frameBuffer,
        m_videoDecoderContext.pOutputInfoOut = &outputInfo;
        m_videoDecoderContext.done = false;
         
        m_videoDecoderContext.conditionVariable.notify_one();
        {
            std::unique_lock<Mutex> lock(m_videoDecoderContext.conditionMutex);
            m_videoDecoderContext.conditionVariable.wait(lock, [&]() { return this->m_videoDecoderContext.done; });
        }

        retVal = m_videoDecoderContext.ret;

    } while (retVal == SCE_VIDEODEC2_ERROR_NEW_SEQUENCE);

    // Fatal error present in stream, therefore the decoder must be reset
    if (retVal == SCE_VIDEODEC2_ERROR_FATAL_STREAM) {
        resetDecoder();
    }

    if (retVal != SCE_OK) {
        if (retVal == SCE_VIDEODEC2_ERROR_OVERSIZE_DECODE) {
            PS4Platform::traceError("sceVideodec2Decode failed - oversize decode", retVal);
            return MediaResult(MediaResult::ErrorNotSupported, retVal);
        }

        PS4Platform::traceError("sceVideodec2Decode failed.", retVal);
        return MediaResult(MediaResult::ErrorInvalidData, retVal);
    }

    // Sometimes, no image is outputted from the decoder. This is not an error.
    if (!outputInfo.isValid) {
        return MediaResult::Ok;
    }

    if (outputInfo.isErrorFrame) {
        TRACE_ERROR("Dropping frame (id=%ld, PTS=%f) due to decoding error", inputSample->id, inputSample->getPresentationTime().seconds());
        return MediaResult::Ok;
    }

    addDecodedOutput(outputInfo, mediaFrameBufferInfo);
    return MediaResult::Ok;
}

MediaResult VideoDecoder::hasOutput(bool& hasOutput)
{
    hasOutput = !m_decodedFrames.empty();
    return MediaResult::Ok;
}

MediaResult VideoDecoder::getOutput(std::shared_ptr<MediaSample>& output)
{
    if (m_decodedFrames.empty()) {
        return MediaResult::ErrorInvalidState;
    } else {
        output = m_decodedFrames.front();
        m_decodedFrames.pop();
        return MediaResult::Ok;
    }
}

MediaResult VideoDecoder::flush()
{
    TRACE_INFO("ps4::VideoDecoder::flush");

    SceVideodec2OutputInfo outputInfo;
    outputInfo.thisSize = sizeof(outputInfo);
    outputInfo.isValid = true;

    MediaResult result = MediaResult::Ok;

    // You must keep calling sceVideodec2Flush until it sets outputInfo.isValid to false
    while (outputInfo.isValid) {
        auto mediaFrameBufferInfo = m_frameBufferPool.getFrameBuffer();
        if (!mediaFrameBufferInfo) {
            // This will happen when the VideoDecoder is trying to destruct or
            // FrameBufferPool times out
            TRACE_WARN("ps4::VideoDecoder::flush(): Resetting the decoder due to full frame buffer");
            result = resetDecoder();
            break;
        }

        SceVideodec2FrameBuffer frameBuffer;
        frameBuffer.thisSize = sizeof(frameBuffer);
        frameBuffer.pFrameBuffer = mediaFrameBufferInfo->pFrame;
        frameBuffer.frameBufferSize = mediaFrameBufferInfo->frameSize;

        int retVal = sceVideodec2Flush(m_decoderHandle,
            &frameBuffer,
            &outputInfo);

        if (retVal) {
            PS4Platform::traceError("sceVideodec2Flush failed.", retVal);
            result = MediaResult(MediaResult::ErrorInvalidOutput, retVal);
            break;
        }

        TRACE_DEBUG("sceVideodec2Flush done isValid:%d", outputInfo.isValid);

        if (outputInfo.isValid) {
            addDecodedOutput(outputInfo, mediaFrameBufferInfo);
        }
    }

    m_decoderBuffer.clear();

    return result;
}

MediaResult VideoDecoder::reset()
{
    TRACE_INFO("ps4::VideoDecoder::reset");
    m_decodedFrames = std::queue<std::shared_ptr<VideoSample>>();
    m_decoderBuffer.clear();
    return resetDecoder();
}

void VideoDecoder::addDecodedOutput(SceVideodec2OutputInfo outputInfo, const std::shared_ptr<MediaFrameBufferInfo>& frameBuffer)
{
    assert(outputInfo.pictureCount > 0);

    auto videoSample = std::make_shared<VideoSample>(outputInfo);
    const auto& pictureInfo = videoSample->getPicture(0);
    assert(pictureInfo.isValid);
    assert(pictureInfo.attachedData > 0);

    auto itr = m_decoderBuffer.find(static_cast<size_t>(pictureInfo.attachedData));
    assert(itr != m_decoderBuffer.end());
    auto decodedInputSample = itr->second;
    m_decoderBuffer.erase(itr);

    assert(pictureInfo.ptsData == (uint64_t)decodedInputSample->mediaSample.presentationTime.nanoseconds().count());

    if (outputInfo.isErrorFrame) {
        TRACE_ERROR("Dropping frame (id=%ld, PTS=%f) due to decoding error", decodedInputSample->id, decodedInputSample->getPresentationTime().seconds());
        return;
    }

    ++m_decodeCount;

    // Discard samples not to be rendered
    if (decodedInputSample->mediaSample.isDecodeOnly) {
        return;
    }

    videoSample->id = ++m_outputCount;
    videoSample->frameBuffer = frameBuffer;
    videoSample->mediaSample = std::move(decodedInputSample->mediaSample);
    m_decodedFrames.push(videoSample);
}

MediaResult VideoDecoder::init()
{
    m_computeQueue = nullptr;
    m_computeQueue.reset(new ComputeQueue());

    if (m_computeQueue->getError()) {
        return MediaResult(MediaResult::ErrorInternal, m_computeQueue->getError());
    }

    return createDecoder();
}

MediaResult VideoDecoder::startDecoding()
{
    if (!m_decoderHandle) {
        return MediaResult::ErrorInvalidState;
    }

    return resetDecoder();
}

bool VideoDecoder::isValidAvcProfile(int avcProfile)
{
    int validProfileLevels[] = {
        SCE_VIDEODEC2_AVC_PROFILE_BASELINE,
        SCE_VIDEODEC2_AVC_PROFILE_MAIN,
        SCE_VIDEODEC2_AVC_PROFILE_HIGH,
    };

    for (int validAvc : validProfileLevels)
        if (avcProfile == validAvc) {
            return true;
        }

    return false;
}

bool VideoDecoder::isValidAvcLevel(int avcLevel)
{
    int validAvcLevels[] = {
        SCE_VIDEODEC2_AVC_LEVEL_1,
        SCE_VIDEODEC2_AVC_LEVEL_1_b,
        SCE_VIDEODEC2_AVC_LEVEL_1_1,
        SCE_VIDEODEC2_AVC_LEVEL_1_2,
        SCE_VIDEODEC2_AVC_LEVEL_1_3,
        SCE_VIDEODEC2_AVC_LEVEL_2,
        SCE_VIDEODEC2_AVC_LEVEL_2_1,
        SCE_VIDEODEC2_AVC_LEVEL_2_2,
        SCE_VIDEODEC2_AVC_LEVEL_3,
        SCE_VIDEODEC2_AVC_LEVEL_3_1,
        SCE_VIDEODEC2_AVC_LEVEL_3_2,
        SCE_VIDEODEC2_AVC_LEVEL_4,
        SCE_VIDEODEC2_AVC_LEVEL_4_1,
        SCE_VIDEODEC2_AVC_LEVEL_4_2,
        SCE_VIDEODEC2_AVC_LEVEL_5,
        SCE_VIDEODEC2_AVC_LEVEL_5_1,
        SCE_VIDEODEC2_AVC_LEVEL_5_2,
        SCE_VIDEODEC2_AVC_LEVEL_1,
    };

    for (int validAvc : validAvcLevels)
        if (avcLevel == validAvc) {
            return true;
        }

    return false;
}

MediaResult VideoDecoder::createDecoder()
{
    if (!m_configuration) {
        TRACE_ERROR("Cannot create decoder without video configuration");
        return MediaResult::ErrorInvalidState;
    }
    int ret = sceVideodec2PresetDecoderConfigInfo(SCE_VIDEODEC2_PRESET_AVC_1080P_HIGH_L4_2_DPB4, m_computeQueue->get(), &m_configInfo);

    if (ret != SCE_OK) {
        PS4Platform::traceError("sceVideodec2PresetDecoderConfigInfo failed", ret);
        return MediaResult(MediaResult::ErrorInvalidParameter, ret);
    }
	
    m_configInfo.maxFrameWidth = m_configuration->width;
    m_configInfo.maxFrameHeight = m_configuration->height;
    m_configInfo.profile = static_cast<uint32_t>(m_configuration->avcProfile);
    m_configInfo.maxLevel = static_cast<uint32_t>(m_configuration->avcLevel);
    m_configInfo.maxDpbFrameCount = m_globalConfiguration.maxDpbFrameCount;

    if (!isValidAvcProfile(m_configInfo.profile)) {
        return MediaResult::ErrorInvalidParameter;
    }

    if (!isValidAvcLevel(m_configInfo.maxLevel)) {
        return MediaResult::ErrorInvalidParameter;
    }

    m_configInfo.decodePipelineDepth = m_globalConfiguration.pipelineDepth; // maximum decoding pipeline (more memory)

    TRACE_DEBUG("Decoder Config Info: Width:%d Height:%d DBP:%d Pipeline:%d Progressive:%d AvcProfile:%d AvcLevel:%d",
        m_configInfo.maxFrameWidth,
        m_configInfo.maxFrameHeight,
        m_configInfo.maxDpbFrameCount,
        m_configInfo.decodePipelineDepth,
        m_configInfo.optimizeProgressiveVideo,
        m_configInfo.profile,
        m_configInfo.maxLevel);

    m_decoderMemoryInfo = nullptr;
    m_decoderMemoryInfo.reset(new DecoderMemoryInfo(m_configInfo));

    if (m_decoderMemoryInfo->getError() != SCE_OK) {
        PS4Platform::traceError("Creating the DecoderMemoryInfo failed", m_decoderMemoryInfo->getError());
        return MediaResult(MediaResult::ErrorInternal, m_decoderMemoryInfo->getError());
    }

    // create the actual decoder
    SceVideodec2Decoder decHandle;
    ret = sceVideodec2CreateDecoder(&m_configInfo,
        &m_decoderMemoryInfo->get(),
        &decHandle);

    if (ret != SCE_OK) {
        m_decoderMemoryInfo = nullptr;
        PS4Platform::traceError("Creating the decoder failed", ret);
        return MediaResult(MediaResult::ErrorInternal, ret);
    }

    TRACE_DEBUG("sceVideodec2CreateDecoder succeeded");

    m_decoderHandle = decHandle;

    return MediaResult(SCE_OK);
}

MediaResult VideoDecoder::destroyDecoder()
{
    if (!m_decoderHandle) {
        return MediaResult::ErrorInvalidState;
    }

    TRACE_DEBUG("Destroying previous decoder.");
    int ret = sceVideodec2DeleteDecoder(m_decoderHandle);

    if (ret < 0) {
        TRACE_DEBUG("sceVideodec2DeleteDecoder failed with error code=0x%08x", ret);
        return MediaResult(MediaResult::ErrorInvalidState, ret);
    }

    m_decoderMemoryInfo = nullptr;
    return MediaResult::Ok;
}

MediaResult VideoDecoder::resetDecoder()
{
    if (!m_decoderHandle) {
        return MediaResult::ErrorInvalidState;
    }

    int ret = sceVideodec2Reset(m_decoderHandle);
    if (ret < 0) {
        PS4Platform::traceError("sceVideodec2Reset failed", ret);
        return MediaResult(MediaResult::ErrorInvalidState, ret);
    }

    return MediaResult::Ok;
}

int VideoDecoder::spawnInternalThread()
{
    ScePthreadAttr attr = 0;
    SceKernelSchedParam schedparam;
    const uint32_t stacksize = 512 * 1024; // "We created an separate thread for sceVideodec2Decode function since this one uses ~57k on stack and 
                                           // "ThreadScheduler" is set at 64k. (https://twitchtv.atlassian.net/browse/CVP-2085)
                                           // https://ps4.siedev.net/support/issue/158862#n2073108

    // create a thread attribute object
    int ret = scePthreadAttrInit(&attr);
    if (ret < 0) {
        TRACE_ERROR("scePthreadAttrInit() failed: 0x%08X", ret);
        return ret;
    }

    // set stack size
    ret = scePthreadAttrSetstacksize(&attr, stacksize);
    if (ret < 0) {
        TRACE_ERROR("scePthreadAttrSetstacksize() failed: 0x%08X", ret);
        return ret;
    }

    // set schedule priority
    ret = scePthreadAttrSetinheritsched(&attr, SCE_PTHREAD_EXPLICIT_SCHED);
    if (ret < 0) {
        TRACE_ERROR("scePthreadAttrSetinheritsched() failed: 0x%08X", ret);
        return ret;
    }

    schedparam.sched_priority = SCE_KERNEL_PRIO_FIFO_HIGHEST;
    ret = scePthreadAttrSetschedparam(&attr, &schedparam);
    if (ret < 0) {
        TRACE_ERROR("scePthreadAttrSetschedparam() failed: 0x%08X", ret);
        return ret;
    }

    // create a thread for video decoding
    ret = scePthreadCreate(&m_videoDecoderContext.decodingThread, &attr, threadVideoDecoder, &m_videoDecoderContext, "Video Decoding");
    if (ret < 0) {
        TRACE_ERROR("scePthreadCreate() failed: 0x%08X", ret);
        return ret;
    }

    ret = scePthreadAttrDestroy(&attr);
    if (ret < 0) {
        TRACE_ERROR("scePthreadAttrDestroy() failed: 0x%08X", ret);
        return ret;
    }

    return ret;
}

void VideoDecoder::releaseInternalThread()
{
    if (m_videoDecoderContext.decodingThread) {
        {
            std::unique_lock<Mutex> lock(m_videoDecoderContext.conditionMutex);
            m_videoDecoderContext.running = false;
        }

        m_videoDecoderContext.conditionVariable.notify_one();

        int threadRet = scePthreadJoin(m_videoDecoderContext.decodingThread, 0);
        m_videoDecoderContext.decodingThread = 0;

        if (threadRet < 0) {
            TRACE_ERROR("scePthreadJoin() failed: 0x%08X", threadRet);
        }
    }
}

int32_t allocateDirectMemory(off_t searchStart, off_t searchEnd,
    size_t len, size_t alignment,
    int memoryType,
    off_t* physAddrOut)
{
    auto error = sceKernelAllocateDirectMemory(searchStart, searchEnd, len, alignment, memoryType, physAddrOut);

    uint32_t maxMem = len;
    uint32_t minMem = 0;
    auto originalError = error;
    while (originalError) {
        // stop searching, we found it
        if (maxMem - minMem <= 32 * 1024)
            break;

        // search the middle
        len = (maxMem - minMem) / 2 + minMem;
        len = roundUp(len, alignment);

        error = sceKernelAllocateDirectMemory(searchStart, searchEnd,
            len, alignment,
            memoryType, physAddrOut);

        if (error) {
            maxMem = len;
        } else {
            minMem = len;
            error = sceKernelReleaseDirectMemory(*physAddrOut, len);
            assert(error == 0);
        }
    }

    if (originalError) {
        TRACE_ERROR("sceKernelAllocateDirectMemory failed (0x%08x). The available memory is %d", originalError, (maxMem + minMem) / 2);
    }

    return originalError;
}

DecoderMemoryInfo::DecoderMemoryInfo(const SceVideodec2DecoderConfigInfo& configInfo)
    : m_cpuGpuOffset(0)
    , m_gpuOnlyOffset(0)
    , m_error(0)
{
    memset(&m_decoderMemoryInfo, 0, sizeof(m_decoderMemoryInfo));

    m_decoderMemoryInfo.thisSize = sizeof(SceVideodec2DecoderMemoryInfo);
    m_error = sceVideodec2QueryDecoderMemoryInfo(&configInfo, &m_decoderMemoryInfo);

    if (m_error != SCE_OK) {
        PS4Platform::traceError("sceVideodec2QueryDecoderMemoryInfo failed", m_error);
        return;
    }

    size_t totalMemorySize = m_decoderMemoryInfo.cpuMemorySize + m_decoderMemoryInfo.gpuMemorySize + m_decoderMemoryInfo.cpuGpuMemorySize;

    TRACE_DEBUG("cpuMemorySize:%3.2fMB gpuMemorySize:%3.2fMB cpuGpuMemorySize:%3.2fMB total:%3.2fMB",
        (float)m_decoderMemoryInfo.cpuMemorySize / 0x100000,
        (float)m_decoderMemoryInfo.gpuMemorySize / 0x100000,
        (float)m_decoderMemoryInfo.cpuGpuMemorySize / 0x100000,
        (float)totalMemorySize / 0x100000);

    TRACE_DEBUG("maxFrameBufferSize:%3.2fMB frameBufferAlignment:%d",
        (float)m_decoderMemoryInfo.maxFrameBufferSize / 0x100000, m_decoderMemoryInfo.frameBufferAlignment);

    void* allocPtr = 0;
    int64_t allocOffset = 0;
    uint32_t totalAlignSize = 0;

    // allocate instance memory for cpuOnly
    if (m_decoderMemoryInfo.cpuMemorySize) {
        allocPtr = malloc(m_decoderMemoryInfo.cpuMemorySize);

        if (!allocPtr) {
            TRACE_DEBUG("malloc, couldn't allocate %d bytes", m_decoderMemoryInfo.cpuMemorySize);
            m_error = SCE_KERNEL_ERROR_EAGAIN;
            return;
        }

        m_decoderMemoryInfo.pCpuMemory = allocPtr;
    }

    allocPtr = nullptr;
    totalAlignSize = roundUp(m_decoderMemoryInfo.cpuGpuMemorySize, SCE_KERNEL_PAGE_SIZE);
    m_decoderMemoryInfo.cpuGpuMemorySize = totalAlignSize;

    static int instanceCount = 0;
    instanceCount++;

    TRACE_DEBUG("Trying to allocate %d bytes of onion memory", totalAlignSize);
    m_error = allocateDirectMemory(0, SCE_KERNEL_MAIN_DMEM_SIZE,
        totalAlignSize, SCE_KERNEL_PAGE_SIZE,
        SCE_KERNEL_WB_ONION, &allocOffset);

    if (m_error) {
        PS4Platform::traceError("allocateDirectMemory failed", m_error);
        return;
    }

    m_cpuGpuOffset = allocOffset;

    TRACE_DEBUG("sceKernelAllocateDirectMemory for CPU/GPU memory done ... size:%d offset:%ld",
        totalAlignSize, allocOffset);

    m_error = sceKernelMapDirectMemory(&allocPtr, totalAlignSize,
        SCE_KERNEL_PROT_CPU_READ | SCE_KERNEL_PROT_CPU_WRITE | SCE_KERNEL_PROT_GPU_ALL,
        0, allocOffset, SCE_KERNEL_PAGE_SIZE);

    if (m_error) {
        PS4Platform::traceError("sceKernelMapDirectMemory failed", m_error);
        return;
    }

    TRACE_DEBUG("sceKernelMapDirectMemory for CPU/GPU memory done ... @0x%p", allocPtr);

    m_decoderMemoryInfo.pCpuGpuMemory = allocPtr;
    allocPtr = nullptr;
    totalAlignSize = roundUp(m_decoderMemoryInfo.gpuMemorySize, SCE_KERNEL_PAGE_SIZE);
    m_decoderMemoryInfo.gpuMemorySize = totalAlignSize;

    TRACE_DEBUG("Trying to allocate %d bytes of garlic memory", totalAlignSize);
    m_error = allocateDirectMemory(0, SCE_KERNEL_MAIN_DMEM_SIZE,
        totalAlignSize, SCE_KERNEL_PAGE_SIZE,
        SCE_KERNEL_WC_GARLIC, &allocOffset);

    if (m_error) {
        PS4Platform::traceError("allocateDirectMemory failed", m_error);
        return;
    }

    m_gpuOnlyOffset = allocOffset;

    TRACE_DEBUG("sceKernelAllocateDirectMemory for GPU only memory done ... size:%d offset:%ld",
        totalAlignSize, allocOffset);

    m_error = sceKernelMapDirectMemory(&allocPtr, totalAlignSize,
        SCE_KERNEL_PROT_CPU_READ | SCE_KERNEL_PROT_CPU_WRITE | SCE_KERNEL_PROT_GPU_ALL,
        0, allocOffset, SCE_KERNEL_PAGE_SIZE);

    if (m_error) {
        PS4Platform::traceError("sceKernelMapDirectMemory failed", m_error);
        return;
    }

    TRACE_DEBUG("sceKernelMapDirectMemory for GPU only memory done ... @0x%p", allocPtr);
    m_decoderMemoryInfo.pGpuMemory = allocPtr;
}

DecoderMemoryInfo::~DecoderMemoryInfo()
{
    if (m_decoderMemoryInfo.pCpuMemory) {
        free(m_decoderMemoryInfo.pCpuMemory);
    }

    int retVal = 0;

    if (m_decoderMemoryInfo.pCpuGpuMemory) {
        retVal = munmap(m_decoderMemoryInfo.pCpuGpuMemory, m_decoderMemoryInfo.cpuGpuMemorySize);

        if (!retVal) {
            sceKernelReleaseDirectMemory(m_cpuGpuOffset, m_decoderMemoryInfo.cpuGpuMemorySize);
        }
    }

    if (m_decoderMemoryInfo.pGpuMemory) {
        retVal = munmap(m_decoderMemoryInfo.pGpuMemory, m_decoderMemoryInfo.gpuMemorySize);

        if (!retVal) {
            sceKernelReleaseDirectMemory(m_gpuOnlyOffset, m_decoderMemoryInfo.gpuMemorySize);
        }
    }
}

const SceVideodec2DecoderMemoryInfo& DecoderMemoryInfo::get() const
{
    return m_decoderMemoryInfo;
}

ComputeQueue::ComputeQueue()
    : m_memoryOffset(0)
    , m_memorySize(0)
    , m_error(0)
{
    allocate(m_computeQueueMemoryInfo, m_computeQueue, m_cfgInfo, m_memoryOffset, m_memorySize);
}

void ComputeQueue::allocate(SceVideodec2ComputeMemoryInfo& computeInfo, SceVideodec2ComputeQueue& computeQueue, SceVideodec2ComputeConfigInfo& cfgInfo, int64_t& memoryOffset, uint32_t& memorySize)
{
    computeInfo.thisSize = sizeof(SceVideodec2ComputeMemoryInfo);

    // Get memory size for compute queue instance.
    m_error = sceVideodec2QueryComputeMemoryInfo(&computeInfo);

    if (m_error != SCE_OK) {
        PS4Platform::traceError("sceVideodec2QueryComputeMemoryInfo failed.", m_error);
        return;
    }

    TRACE_DEBUG("sceVideodec2QueryComputeMemoryInfo done ... cpuGpuMemorySize:%zd",
        computeInfo.cpuGpuMemorySize);

    // Direct memory size must be aligned by page size, at least.
    int alignedMemSize = ~(SCE_KERNEL_PAGE_SIZE - 1) & (computeInfo.cpuGpuMemorySize + SCE_KERNEL_PAGE_SIZE - 1);

    int64_t memOffset = 0;
    TRACE_DEBUG("Trying to allocate %d bytes of onion memory.", alignedMemSize);
    m_error = allocateDirectMemory(0, SCE_KERNEL_MAIN_DMEM_SIZE,
        alignedMemSize, SCE_KERNEL_PAGE_SIZE,
        SCE_KERNEL_WB_ONION, &memOffset);

    if (m_error != SCE_OK) {
        PS4Platform::traceError("sceKernelAllocateDirectMemory failed", m_error);
        return;
    }

    TRACE_DEBUG("sceKernelAllocateDirectMemory done ... size:%d offset:%ld",
        alignedMemSize, memOffset);

    memoryOffset = memOffset;
    memorySize = alignedMemSize;

    void* pMem = nullptr;
    m_error = sceKernelMapDirectMemory(&pMem, alignedMemSize,
        (SCE_KERNEL_PROT_CPU_RW | SCE_KERNEL_PROT_GPU_RW),
        0, memOffset, SCE_KERNEL_PAGE_SIZE);

    if (m_error != SCE_OK) {
        PS4Platform::traceError("sceKernelMapDirectMemory failed", m_error);
        releaseMemory(memoryOffset, memorySize);
        return;
    }

    TRACE_DEBUG("sceKernelMapDirectMemory done ... CPU/GPU memory @0x%p", pMem);
    computeInfo.pCpuGpuMemory = pMem;

    cfgInfo.thisSize = sizeof(SceVideodec2ComputeConfigInfo);

    cfgInfo.computePipeId = cfgInfo.computeQueueId = m_computeQueueManager.getNextFree();
    cfgInfo.checkMemoryType = true;
    cfgInfo.reserved0 = 0;
    cfgInfo.reserved1 = 0;

    // Allocate compute queue
    m_error = sceVideodec2AllocateComputeQueue(&cfgInfo, &computeInfo, &computeQueue);

    if (m_error != SCE_OK) {
        PS4Platform::traceError("sceVideodec2AllocateComputeQueue failed.", m_error);
        releaseMemory(memoryOffset, memorySize);
        return;
    }

    TRACE_DEBUG("sceVideodec2AllocateComputeQueue returned SCE_OK.");
}

ComputeQueue::~ComputeQueue()
{
    releaseQueue(m_computeQueue, m_cfgInfo);
    releaseMemory(m_memoryOffset, m_memorySize);
}

void ComputeQueue::releaseQueue(SceVideodec2ComputeQueue& computeQueue, SceVideodec2ComputeConfigInfo& cfgInfo)
{
    if (computeQueue) {
        m_error = sceVideodec2ReleaseComputeQueue(computeQueue);
        computeQueue = nullptr;
        assert(cfgInfo.computePipeId == cfgInfo.computeQueueId);
        bool success = m_computeQueueManager.release(cfgInfo.computePipeId);
        if (!success) {
            TRACE_DEBUG("ComputeQueue Manager was unable to release pipe %d", cfgInfo.computePipeId);
        }
    }

    TRACE_DEBUG("ComputeQueue sceVideodec2ReleaseComputeQueue completed");
}

void ComputeQueue::releaseMemory(int64_t& memoryOffset, uint32_t& memorySize)
{
    if (memoryOffset && memorySize) {
        m_error = sceKernelReleaseDirectMemory(memoryOffset, memorySize);
        memoryOffset = 0;
        memorySize = 0;
    }

    TRACE_DEBUG("ComputeQueue sceKernelReleaseDirectMemory completed");
}

int HardwareComputePipeQueueManager::getNextFree()
{
    std::lock_guard<Mutex> lock(m_mutex);
    for (int i = 0; i < MaxPipe; ++i) {
        if (!m_used[i]) {
            m_used[i] = true;
            return i;
        }
    }

    return -1;
}

bool HardwareComputePipeQueueManager::release(int handle)
{
    std::lock_guard<Mutex> lock(m_mutex);

    if (handle < 0 || handle >= MaxPipe) {
        return false;
    }

    if (!m_used[handle]) {
        return false;
    }

    m_used[handle] = false;
    return true;
}

bool HardwareComputePipeQueueManager::m_used[] = { false, false, false, false, false };

Mutex HardwareComputePipeQueueManager::m_mutex;
