#pragma once

#include "FrameBufferPool.hpp"
#include "playercore/MediaDecoder.hpp"
#include "MediaFrameBuffer.hpp"
#include "playercore/platform/ps4/Configuration.hpp"
#include "platforms/ps4/MediaFrameBuffer.hpp"
#include "VideoSample.hpp"
#include <videodec2.h>
#include <memory>
#include <queue>
#include <unordered_map>

namespace twitch {
namespace ps4 {
    class FrameBufferPool;

    // wrapper class over SceVideodec2DecoderMemoryInfo
    class DecoderMemoryInfo {
    public:
        DecoderMemoryInfo(const SceVideodec2DecoderConfigInfo& configInfo);
        ~DecoderMemoryInfo();

        const SceVideodec2DecoderMemoryInfo& get() const;
        int getError() const { return m_error; }

    private:
        SceVideodec2DecoderMemoryInfo m_decoderMemoryInfo;
        int64_t m_cpuGpuOffset;
        int64_t m_gpuOnlyOffset;

        int m_error;
    };

    class HardwareComputePipeQueueManager {
    public:
        int getNextFree();
        bool release(int handle);

    private:
        static Mutex m_mutex;
        static const int MaxPipe = 5;
        static bool m_used[MaxPipe];
    };

    // wrapper class of SceVideodec2ComputeQueue and SceVideodec2ComputeMemoryInfo
    class ComputeQueue {
    public:
        ComputeQueue();
        ~ComputeQueue();

        int getError() const { return m_error; }

        const SceVideodec2ComputeQueue& get() const { return m_computeQueue; }

    protected:
    private:
        void allocate(SceVideodec2ComputeMemoryInfo& computeInfo, SceVideodec2ComputeQueue& computeQueue, SceVideodec2ComputeConfigInfo& cfgInfo, int64_t& memoryOffset, uint32_t& memorySize);
        void releaseQueue(SceVideodec2ComputeQueue& computeQueue, SceVideodec2ComputeConfigInfo& cfgInfo);
        void releaseMemory(int64_t& memoryOffset, uint32_t& memorySize);

        HardwareComputePipeQueueManager m_computeQueueManager;

        SceVideodec2ComputeMemoryInfo m_computeQueueMemoryInfo;
        SceVideodec2ComputeQueue m_computeQueue;
        SceVideodec2ComputeConfigInfo m_cfgInfo;
        int64_t m_memoryOffset;
        uint32_t m_memorySize;
        int m_error;
    };

    class VideoDecoder : public MediaDecoder {
    public:
        struct Configuration {
            static std::unique_ptr<Configuration> create(const MediaFormat& format);

            Configuration(int width, int height, int avcLevel, int avcProfile);

            bool operator==(const Configuration& other) const;
            bool isValid(ps4::Configuration & validator) const;

            int width = -1;
            int height = -1;
            int avcLevel = -1;
            int avcProfile = -1;
        };

        VideoDecoder(const ps4::Configuration & globalConfiguration);
        virtual ~VideoDecoder();

        MediaResult configure(const MediaFormat& input, MediaFormat& output) override;
        MediaResult decode(const MediaSampleBuffer& input) override;
        MediaResult hasOutput(bool& hasOutput) override;
        MediaResult getOutput(std::shared_ptr<MediaSample>& output) override;
        MediaResult flush() override;
        MediaResult reset() override;
    private:
        MediaResult m_ctorResult = MediaResult::Ok;

        MediaResult init();

        MediaResult startDecoding();
        static bool isValidAvcLevel(int avcLevel);
        static bool isValidAvcProfile(int avcProfile);

        MediaResult configureDecoder();
        MediaResult createDecoder();
        MediaResult destroyDecoder();
        MediaResult resetDecoder();

        void addDecodedOutput(SceVideodec2OutputInfo outputInfo, const std::shared_ptr<MediaFrameBufferInfo>& frameBuffer);

        int spawnInternalThread();
        static void* threadVideoDecoder(void* arg);
        void releaseInternalThread();

        ps4::Configuration m_globalConfiguration;
        std::unique_ptr<Configuration> m_configuration;

        std::queue<std::shared_ptr<VideoSample>> m_decodedFrames;

        // decoder stuff
        std::unique_ptr<DecoderMemoryInfo> m_decoderMemoryInfo;
        std::unique_ptr<ComputeQueue> m_computeQueue;

        SceVideodec2DecoderConfigInfo m_configInfo;
        SceVideodec2Decoder m_decoderHandle;

        FrameBufferPool m_frameBufferPool;

        struct InputSample {
            size_t id;
            MediaSampleBuffer mediaSample;

            const MediaTime& getPresentationTime() const { return mediaSample.presentationTime; }
        };

        // Buffer of samples that are currently used by the decoder
        std::unordered_map<size_t, std::shared_ptr<InputSample>> m_decoderBuffer;

        size_t m_inputCount = 0;
        size_t m_decodeCount = 0;
        size_t m_outputCount = 0;

        struct VideoDecoderContext
        {
            SceVideodec2Decoder decoderInstanceIn;
            const SceVideodec2InputData *pInputDataInOut;
            SceVideodec2FrameBuffer *pFrameBufferInOut;
            SceVideodec2OutputInfo *pOutputInfoOut;

            int32_t ret;
            volatile bool done;
            volatile bool running = true;

            ScePthread decodingThread = 0;
            ConditionVariable conditionVariable;
            Mutex conditionMutex;
        };

        VideoDecoderContext m_videoDecoderContext;
    };
}
}
