#include "VideoFrameBuffer.hpp"
#include "ProcessClock.hpp"
#include "debug/trace.hpp"
#include "VideoSample.hpp"
#include <cassert>

#include "playercore/platform/ps4/PS4Platform.hpp"

namespace twitch {
namespace ps4 {

    VideoFrameBuffer::VideoFrameBuffer(Listener& listener, const ReferenceClock& clock, int frameBufferCount)
        : m_listener(listener)
        , m_clock(clock)
        , m_timeToRender(10)
        , m_overrideSeekTime(MediaTime::invalid())
    {
        // Make sure our buffer size is smaller than the buffer pool's
        m_maxFramesSize = frameBufferCount - 2;
        assert(frameBufferCount > 2);
    }

    void VideoFrameBuffer::addFrame(const std::shared_ptr<const VideoSample>& sample)
    {
        auto frame = std::make_shared<VideoFrame>(*this, sample);

        std::unique_lock<Mutex> lock(m_mutex);

        // Verify that the sample PTS is increasing
        auto lastFrame = m_frames.empty() ? m_current : m_frames.back();
        if (lastFrame) {
            // assert(m_frames.back()->getPresentationTime() < sample->getPresentationTime());
            // FIXME: handle edge condition where a discontinuity (i.e. bitrate switch) will
            //   decreases in PTS
            if (lastFrame->getPresentationTime() > sample->getPresentationTime()) {
                TRACE_DEBUG("VideoFrameBuffer::addFrame(): Dropping frame at PTS=%f due to PTS moving backwards (previous=%f)", sample->getPresentationTime().seconds(), lastFrame->getPresentationTime().seconds());
                m_listener.onDroppedFrame();
                return;
            }
        }

        // Check if we need to drop frames based on the clock and PTS
        auto elapsedPts = m_clock.getMediaTime();
        if (!elapsedPts.valid()) {
            elapsedPts = MediaTime::zero();
        }
        if (!m_frames.empty()) {
            // Loop over the frames to make sure that the next frame in the buffer
            // should be the next frame rendered
            auto currFrame = m_frames.begin();
            auto nextFrame = currFrame + 1;
            for (; nextFrame != m_frames.end(); ++currFrame, ++nextFrame) {
                // Check if the targetPts is closer to the next frame
                auto midpointPts = ((*currFrame)->getPresentationTime() + (*nextFrame)->getPresentationTime()) / 2;
                if (elapsedPts >= midpointPts) {
                    TRACE_INFO("VideoFrameBuffer::addFrame(): Dropping frame at PTS=%f since current renderer clock is at PTS=%f", (*currFrame)->getPresentationTime().seconds(), elapsedPts.seconds());
                    m_listener.onDroppedFrame();
                    m_frames.pop_front();
                }
                else {
                    break;
                }
            }
        }

        // Check if the buffer has reached capacity
        if (m_frames.size() >= m_maxFramesSize) {
            // Wait to see if the UI will consume a frame to make room on the buffer
            // for the new frame
            std::chrono::microseconds timeout(100000);
            auto waitReturn = m_condition.wait_for(lock, timeout);
            if (waitReturn == CvStatus::timeout && !m_frames.empty()) {
                const auto& droppedFrame = m_frames.front();
                TRACE_INFO("VideoFrameBuffer::addFrame(): Dropping frame at PTS=%f due to full buffer. elapsedPts=%f",
                    droppedFrame->getPresentationTime().seconds(),
                    elapsedPts.seconds()
                );
                m_listener.onDroppedFrame();

                // Seek forward fix. In case where we are seeking forward, the audio track will be paused.
                // The video frame buffer will be full, and m_current will not advance anymore.
                // Advance m_current to the next value in the video frame buffer to unblock it.
                // Ideally, this code would only be run during a seek
                bool assignNext = m_current && m_current->isFinished() && m_current->getPresentationTime() < m_frames.front()->getPresentationTime();
                m_frames.pop_front();
                if (!m_frames.empty() && assignNext) {
                    m_overrideSeekTime = m_frames.front()->getPresentationTime();
                }
            }
        }

        assert(m_frames.size() <= m_maxFramesSize);
        m_frames.push_back(frame);
    }

    void VideoFrameBuffer::flush()
    {
        TRACE_DEBUG("VideoFrameBuffer::flush()");
        // Release all of the frames back to memory
        std::lock_guard<Mutex> lock(m_mutex);

        for (auto& frame : m_frames) {
            m_listener.onFrameNeedsReleasing(frame);
        }

        if (m_current) {
            m_listener.onFrameNeedsReleasing(m_current);
        }

        m_frames.clear();
        m_condition.notify_all();
        if (m_current) {
            // Don't allow callback to update render time
            m_current->flush();
            m_current.reset();
        }
    }

    std::shared_ptr<twitch::VideoFrame> VideoFrameBuffer::getFrame()
    {
        std::lock_guard<Mutex> lock(m_mutex);
        if (m_frames.empty()) {
            return std::shared_ptr<twitch::VideoFrame>();
        }

        // Estimate the next frame that needs to be rendered
        auto now = ProcessClock::now();

        MediaTime targetPts = m_overrideSeekTime.valid() ? m_overrideSeekTime : m_clock.getMediaTime();
        if (!targetPts.valid()) {
            targetPts = MediaTime::zero();
        }
        m_overrideSeekTime = MediaTime::invalid();

        // Consider frame rate when picking the next sample to render
        if (m_timeToRender.samples() > 0) {
            targetPts += MediaTime(std::chrono::microseconds(m_timeToRender.average()));
        }

        // Find the next frame that should be rendered, starting with the current frame
        if (m_current == nullptr) {
            m_current = m_frames.front();
            m_frames.pop_front();
            m_condition.notify_one();
        }

        int numDropped = 0;
        while (!m_frames.empty()) {
            auto next = m_frames.front();

            // Hysteresis protection. Do not drop a single frame if:
            // 1) It would be the only frame to drop AND
            // 2) We are not passed the midpoint of the next frame AND
            // 3) We haven't already been dropping previous frames (we want to sync to the correct one then)

            // Check if the next frame is a better choice
            if (targetPts >= next->getPresentationTime()) {
                if (!m_current->isFinished()) {

                    bool dropIt = true;
                    // Check criteria 3
                    if (!numDropped) {
                        // Check criteria 1
                        if (m_frames.size() > 1) {
                            auto& subNext = m_frames.at(1);
                            if (targetPts < subNext->getPresentationTime()) {
                                // Check criteria 2
                                if (targetPts < next->getPresentationTime() + (next->getDuration() / 2)) {
                                    dropIt = false;
                                    auto currSample = m_current->getSample();
                                    //TRACE_INFO("VideoFrameBuffer::getFrame(): Not dropping frame at PTS=%f even if current renderer clock is at PTS=%f because next frame is at PTS=%f and the one after is at PTS=%f",
                                    //    currSample.mediaSample.presentationTime.seconds(), targetPts.seconds(), next->getPresentationTime().seconds(), subNext->getPresentationTime().seconds());
                                }
                            }
                        }
                    }

                    if (dropIt) {
                        numDropped++;
                        auto currSample = m_current->getSample();
                        TRACE_INFO("VideoFrameBuffer::getFrame(): Dropping frame at PTS=%f since current renderer clock is at PTS=%f", currSample.mediaSample.presentationTime.seconds(), targetPts.seconds());
                        m_listener.onDroppedFrame();
                    } else {
                        break;
                    }
                }

                m_current = next;
                m_frames.pop_front();
                m_condition.notify_one();
            } else {
                break;
            }
        }

        m_current->setRenderStartTime(now);
        return m_current;
    }

    void VideoFrameBuffer::onFrameRendered(const VideoFrame& frame)
    {
        auto now = ProcessClock::now();
        auto timeToRender = now - frame.getRenderStartTime();
        m_timeToRender.add(timeToRender.microseconds().count());

        std::lock_guard<Mutex> lock(m_mutex);
        if (!m_current) {
            TRACE_DEBUG("VideoFrameBuffer::onFrameRendered(): Skipping render time update due to buffer flush");
            return;
        }
        assert(m_current.get() == &frame);
        m_listener.onFrameRendered(frame.getPresentationTime());
    }

    void VideoFrameBuffer::onFrameNeedsReleasing(std::shared_ptr<VideoFrame>& videoFrame)
    {
        m_listener.onFrameNeedsReleasing(videoFrame);
    }

    void VideoFrameBuffer::start()
    {
    }

    void VideoFrameBuffer::stop()
    {
    }
}
}
