#include "VideoRenderer.hpp"
#include "debug/trace.hpp"
#include "VideoFrame.hpp"
#include "VideoSample.hpp"

namespace twitch {
namespace ps4 {

    VideoRenderer::VideoRenderer(PlayerCore::OnVideoFrameBuffer onVideoFrameBuffer, PlayerCore::OnFrameNeedsReleasing onFrameNeedsReleasing, const ReferenceClock& clock, int frameBufferCount)
    : m_frameBuffer(std::make_shared<VideoFrameBuffer>(*this, clock, frameBufferCount))
    , m_onFrameNeedsReleasing(onFrameNeedsReleasing)
    {
        if (onVideoFrameBuffer) {
            onVideoFrameBuffer(m_frameBuffer);
        } else {
            TRACE_WARN("VideoRenderer: onVideoFrameBuffer not set. Nothing will be visible.");
        }
    }

    VideoRenderer::~VideoRenderer()
    {
        TRACE_DEBUG("~VideoRenderer()");
        m_frameBuffer->flush();
        m_frameBuffer->stop();
    }

    MediaResult VideoRenderer::configure(const MediaFormat& format)
    {
        if (!m_onFrameNeedsReleasing) {
            TRACE_ERROR("VideoRenderer no onFrameNeedsReleasing callback set");
            return MediaResult::ErrorInvalidState;
        }
        (void)format;
        return MediaResult::Ok;
    }

    MediaResult VideoRenderer::getDroppedFrames(int& count)
    {
        count = m_droppedFrames;
        return MediaResult::Ok;
    }

    MediaResult VideoRenderer::getRenderedFrames(int& count)
    {
        count = m_renderedFrames;
        return MediaResult::Ok;
    }

    MediaResult VideoRenderer::getRenderedPresentationTime(MediaTime& time)
    {
        time = m_lastPresentationTime;
        return MediaResult::Ok;
    }

    MediaResult VideoRenderer::render(std::shared_ptr<const MediaSample> input)
    {
        if (input->type != MediaSample::Type::Opaque) {
            return MediaResult::ErrorInvalidData;
        }

        auto videoSample = std::dynamic_pointer_cast<const VideoSample>(input);
        if (!videoSample) {
            return MediaResult::ErrorInvalidData;
        }

        m_frameBuffer->addFrame(videoSample);
        return MediaResult::Ok;
    }

    MediaResult VideoRenderer::flush()
    {
        m_droppedFrames = 0;
        m_renderedFrames = 0;
        m_frameBuffer->flush();
        return MediaResult::Ok;
    }

    MediaResult VideoRenderer::start()
    {
        m_frameBuffer->start();
        return MediaResult::Ok;
    }

    MediaResult VideoRenderer::stop()
    {
        m_frameBuffer->stop();
        return MediaResult::Ok;
    }

    MediaResult VideoRenderer::setPlaybackRate(float rate)
    {
        return (rate != 1.0f) ? MediaResult::ErrorNotSupported : MediaResult::Ok;
    }

    void VideoRenderer::onFrameRendered(const MediaTime& presentationTime)
    {
        m_renderedFrames++;
        m_lastPresentationTime = presentationTime;
    }

    void VideoRenderer::onFrameNeedsReleasing(std::shared_ptr<VideoFrame>& videoFrame)
    {
        if (m_onFrameNeedsReleasing) {
            m_onFrameNeedsReleasing(videoFrame);
        }
    }

    void VideoRenderer::onDroppedFrame()
    {
        m_droppedFrames++;
    }

    void VideoRenderer::dumpAvcInformation(const SceVideodec2AvcPictureInfo& avcPicInfo, int decodeCount) const
    {
        TRACE_DEBUG("#%d PTS:%ld DTS:%ld Attached:%ld %s",
            decodeCount,
            avcPicInfo.ptsData,
            avcPicInfo.dtsData,
            avcPicInfo.attachedData,
            avcPicInfo.idrPictureflag ? "IDR" : "non-IDR");
        TRACE_DEBUG("#%d profile_idc:%d level_idc:%d frame_mbs_only_flag:%d",
            decodeCount,
            avcPicInfo.profile_idc,
            avcPicInfo.level_idc,
            avcPicInfo.frame_mbs_only_flag);

        if (avcPicInfo.bitstream_restriction_flag) {
            TRACE_DEBUG("#%d bitstream_restriction_flag:%d max_dec_frame_buffering:%d",
                decodeCount,
                avcPicInfo.bitstream_restriction_flag,
                avcPicInfo.max_dec_frame_buffering);
        }

        TRACE_DEBUG("#%d pic_width_in_mbs:%d(%d) pic_height_in_map_units:%d(%d)",
            decodeCount,
            avcPicInfo.pic_width_in_mbs_minus1 + 1,
            (avcPicInfo.pic_width_in_mbs_minus1 + 1) * 16,
            avcPicInfo.pic_height_in_map_units_minus1 + 1,
            (avcPicInfo.pic_height_in_map_units_minus1 + 1) * 16);

        if (avcPicInfo.frame_cropping_flag) {
            TRACE_DEBUG("#%d frame_cropping_flag:%d ... top:%d bottom:%d left:%d right:%d",
                decodeCount,
                avcPicInfo.frame_cropping_flag,
                avcPicInfo.frameCropLeftOffset,
                avcPicInfo.frameCropRightOffset,
                avcPicInfo.frameCropTopOffset,
                avcPicInfo.frameCropBottomOffset);
        }

        if (avcPicInfo.video_signal_type_present_flag) {
            TRACE_DEBUG("#%d video_signal_type_present:%d video_format:%d video_full_range_flag:%d",
                decodeCount,
                avcPicInfo.video_signal_type_present_flag,
                avcPicInfo.video_format,
                avcPicInfo.video_full_range_flag);

            if (avcPicInfo.colour_description_present_flag) {
                TRACE_DEBUG("#%d colour_primaries:%d transfer_characteristics:%d matrix_coefficients:%d",
                    decodeCount,
                    avcPicInfo.colour_primaries,
                    avcPicInfo.transfer_characteristics,
                    avcPicInfo.matrix_coefficients);
            }
        }

        if (avcPicInfo.pic_struct_present_flag) {
            TRACE_DEBUG("#%d pic_struct_present:%d pic_struct:%d field_pic_flag:%d bottom_field_flag:%d",
                decodeCount,
                avcPicInfo.pic_struct_present_flag,
                avcPicInfo.pic_struct,
                avcPicInfo.field_pic_flag,
                avcPicInfo.bottom_field_flag);
        }

        if (avcPicInfo.aspect_ratio_info_present_flag) {
            TRACE_DEBUG("#%d aspect_ratio_info_present:%d aspect_ratio_idc:%d sar_width:%d sar_height:%d",
                decodeCount,
                avcPicInfo.aspect_ratio_info_present_flag,
                avcPicInfo.aspect_ratio_idc,
                avcPicInfo.sar_width,
                avcPicInfo.sar_height);
        }

        if (avcPicInfo.timing_info_present_flag) {
            TRACE_DEBUG("#%d timing_info_present:%d num_units_in_tick:%d time_scale:%d fixed_frame_rate_flag:%d",
                decodeCount,
                avcPicInfo.timing_info_present_flag,
                avcPicInfo.num_units_in_tick,
                avcPicInfo.time_scale,
                avcPicInfo.fixed_frame_rate_flag);
        }
    }

    float VideoRenderer::getAvcFrameRate(const SceVideodec2AvcPictureInfo& pPicInfo)
    {
        float frameRate;
        float numScaleUnit = 2.0;

        if (pPicInfo.timing_info_present_flag) {
            if (pPicInfo.fixed_frame_rate_flag) {
                frameRate = (float)pPicInfo.time_scale / pPicInfo.num_units_in_tick;
                frameRate /= numScaleUnit;

                //TRACE_DEBUG("calculated frame rate ... %5.2f fps", frameRate);
            } else { // this sample does not support variable frame rate.
                // set default frame rate.
                frameRate = 60;
            }
        } else {
            // set default frame rate.
            frameRate = 60;
        }

        return frameRate;
    }
}
}
