#include "pch.h"
#include "playercore/platform/windows/WindowsPlatform.hpp"
#include "debug/trace.hpp"

#include "AudioRenderer.hpp"
#include "XAudioBuffer.hpp"
#include "XMasterVoice.hpp"

using namespace twitch;
using namespace twitch::uwp;

using namespace Microsoft;
using namespace Microsoft::WRL;

#define AUDIO_RENDERER_TIME_DEBUG 0

std::unique_ptr<uwp::AudioRenderer> uwp::AudioRenderer::create()
{
    TRACE_DEBUG("AudioRenderer::create()");

    Microsoft::WRL::ComPtr<IXAudio2> xaudio2;
    HRESULT hr = XAudio2Create(&xaudio2, 0, XAUDIO2_DEFAULT_PROCESSOR);

    if (FAILED(hr)) {
        WindowsPlatform::hError("XAudio2Create failed.", hr);
        return nullptr;
    }

    auto masterVoice = XMasterVoice::create(xaudio2);
    if (!masterVoice) {
        TRACE_ERROR("Can't create AudioRenderer without XMasterVoice");
        return nullptr;
    }

    return std::unique_ptr<uwp::AudioRenderer>(new uwp::AudioRenderer(xaudio2, masterVoice));
}

uwp::AudioRenderer::AudioRenderer(Microsoft::WRL::ComPtr<IXAudio2> xaudio2, std::shared_ptr<XMasterVoice> masterVoice)
    : CCoInitialize()
    , m_xAudio2(xaudio2)
    , m_masterVoice(masterVoice)
    , m_threadId(std::this_thread::get_id())
    , m_monitor(std::unique_ptr<XAudioMonitor>(new XAudioMonitor(m_xAudio2)))
{
    TRACE_DEBUG("AudioRenderer::Ctor");
}

uwp::AudioRenderer::~AudioRenderer()
{
    TRACE_DEBUG("AudioRenderer::DTor");
    m_masterVoice.reset();
}

MediaResult uwp::AudioRenderer::configure(const MediaFormat& format)
{
    TRACE_DEBUG("AudioRenderer::configure() called.");
    WAVEFORMATEX wfx;

    const WORD numChannels = format.hasInt(MediaFormat::Audio_ChannelCount) ? static_cast<WORD>(format.getInt(MediaFormat::Audio_ChannelCount)) : 2;
    const int samplesPerSecond = format.hasInt(MediaFormat::Audio_SampleRate) ? format.getInt(MediaFormat::Audio_SampleRate) : 48000;
    const WORD bitsPerSample = format.hasInt(MediaFormat::Audio_BitSize) ? static_cast<WORD>(format.getInt(MediaFormat::Audio_BitSize)) : 16;

    if (m_numChannels == numChannels && m_samplesPerSecond == samplesPerSecond && m_bitsPerSample == bitsPerSample) {
        TRACE_INFO("AudioRenderer::configure - ignored since settings are the same.");
        return MediaResult::Ok;
    }

    TRACE_INFO("AudioRenderer::configure - configuring renderer.");

    m_state = State::Stopped;
    m_numChannels = numChannels;
    m_samplesPerSecond = samplesPerSecond;
    m_bitsPerSample = bitsPerSample;

    wfx.wFormatTag = WAVE_FORMAT_PCM;
    wfx.nChannels = numChannels;
    wfx.nSamplesPerSec = samplesPerSecond;
    wfx.wBitsPerSample = bitsPerSample;
    wfx.nBlockAlign = wfx.nChannels * wfx.wBitsPerSample / 8;
    wfx.nAvgBytesPerSec = wfx.nSamplesPerSec * wfx.wBitsPerSample * wfx.nChannels / 8;
    wfx.cbSize = 0;

    HRESULT hr = m_sourceVoice.configure(m_xAudio2, wfx);
    if (FAILED(hr)) {
        WindowsPlatform::hError("XAudio2::CreateSourceVoice failed.", hr);
        return MediaResult(MediaResult::ErrorInvalidParameter, hr);
    }

    return MediaResult::Ok;
}

MediaResult uwp::AudioRenderer::render(std::shared_ptr<const twitch::MediaSample> input)
{
    if (m_state != State::Started) {
        TRACE_DEBUG("AudioRenderer::start() has not yet been called while in AudioRenderer::render, calling it automatically.");
        start();
    }

    if (m_state != State::Started) {
        return MediaResult(MediaResult::ErrorInvalidState, static_cast<int>(m_state));
    }

    HRESULT result = m_sourceVoice.submit(static_cast<const windows::AudioSample&>(*input));
    if (FAILED(result)) {
        WindowsPlatform::hError("XAudio2 - Could not submit voice buffer", result);
        return MediaResult(MediaResult::ErrorInvalidParameter, result);
    }

    m_monitor->update(*input);

    return MediaResult::Ok;
}

MediaResult uwp::AudioRenderer::getRenderedPresentationTime(MediaTime& time)
{
    if (m_state == State::Stopped) {
        time = MediaTime::invalid();
        return MediaResult::Ok;
    }

    if (!m_sourceVoice.getRenderedPresentationTime(time)) {
        TRACE_ERROR("XAudio2::SourceVoice::getRenderedPresentationTime failed");
        return MediaResult::ErrorInvalidState;
    }

#if AUDIO_RENDERER_TIME_DEBUG
    // Convert all times to ms because that's easier to read
    using namespace std::chrono;
    auto elapsedMs = duration_cast<milliseconds>(std::chrono::system_clock::now() - m_startTime);
    auto timeMs = duration_cast<milliseconds>(time);
    long long deltaMs = timeMs.count() - elapsedMs.count();

    TRACE_DEBUG("AudioRenderer::getRenderedPresentationTime - Elapsed Time: %I64d ms, MediaTime: %I64d ms, delta: %I64d ms",
        elapsedMs.count(), timeMs.count(), deltaMs);
#endif

    return MediaResult::Ok;
}

twitch::MediaResult uwp::AudioRenderer::start()
{
    TRACE_DEBUG("AudioRenderer::start() called.");
    // Already started, ignore it
    if (m_state == State::Started) {
        return MediaResult::Ok;
    }

    // This will make sound starts playing as soon as buffers are submitted
    HRESULT hr = m_sourceVoice.start();
    if (FAILED(hr)) {
        WindowsPlatform::hError("XAudio2::SourceVoice::Start failed.", hr);
        return MediaResult(MediaResult::ErrorInvalidState, hr);
    }

    m_startTime = std::chrono::system_clock::now();
    m_state = State::Started;

    return MediaResult::Ok;
}

twitch::MediaResult uwp::AudioRenderer::stop()
{
    TRACE_DEBUG("AudioRenderer::stop() called.");
    // Already stopped, ignore it
    if (m_state == State::Stopped) {
        return MediaResult::Ok;
    }

    HRESULT hr = m_sourceVoice.stop();
    if (FAILED(hr)) {
        WindowsPlatform::hError("XAudio2::SourceVoice::Stop failed.", hr);
        return MediaResult(MediaResult::ErrorInvalidState, hr);
    }

    m_state = State::Stopped;

    return MediaResult::Ok;
}

twitch::MediaResult uwp::AudioRenderer::flush()
{
    TRACE_DEBUG("AudioRenderer::flush() called.");
    HRESULT hr = m_sourceVoice.flush();
    if (FAILED(hr)) {
        WindowsPlatform::hError("XAudio2::SourceVoice::Flush failed.", hr);
        return MediaResult(MediaResult::Error, hr);
    }

    return MediaResult::Ok;
}

twitch::MediaResult uwp::AudioRenderer::setPlaybackRate(float rate)
{
    m_sourceVoice.setPlaybackRate(rate);
    return MediaResult::Ok;
}

twitch::MediaResult uwp::AudioRenderer::setVolume(float volume)
{
    TRACE_DEBUG("AudioRenderer::setVolume(%f) called.", volume);
    m_sourceVoice.setVolume(volume);
    return MediaResult::Ok;
}
