#include "pch.h"
#include "HttpClient.hpp"
#include "HttpRequest.hpp"
#include "StringUtil.hpp"

using namespace Concurrency;
using namespace Platform;
using namespace Windows::Foundation::Collections;
using namespace Windows::Storage::Streams;
using namespace Windows::Web::Http;

namespace twitch {
namespace uwp {
static const int HttpBufferSize = 16384;

HttpClient::HttpClient()
{
    m_client = ref new Windows::Web::Http::HttpClient();
    m_client->DefaultRequestHeaders->UserAgent->ParseAdd("PlayerCore (UWP)");
}

std::shared_ptr<twitch::HttpRequest> HttpClient::createRequest(const std::string& url, twitch::HttpMethod method)
{
    return std::shared_ptr<twitch::HttpRequest>(new uwp::HttpRequest(m_client, url, method));
}

void HttpClient::send(std::shared_ptr<twitch::HttpRequest> request, ResponseHandler onResponse, ErrorHandler onError)
{
    auto uwpRequest = std::static_pointer_cast<twitch::uwp::HttpRequest>(request);
    uwpRequest->send(onResponse, onError);
}

HttpResponse::HttpResponse(HttpResponseMessage ^ responseMessage
    , int syncPoint
)
    : twitch::HttpResponse(static_cast<int>(responseMessage->StatusCode))
    , m_responseMessage(responseMessage)
    , m_syncPoint(syncPoint)
{
    m_cancelCallback = m_cts.get_token().register_callback(std::bind(&HttpResponse::onCancel, this));
}

HttpResponse::~HttpResponse()
{
    m_cts.get_token().deregister_callback(m_cancelCallback);

    if (m_timer) {
        m_timer->stop();
    }
}

std::string HttpResponse::getHeader(const std::string& key) const
{
    String^ keyRequested = StringUtil::toString(key);

    if (m_responseMessage->Headers->HasKey(keyRequested)) {
        String^ value = m_responseMessage->Headers->Lookup(keyRequested);
        return StringUtil::toString(value);
    }

    if (m_responseMessage->Content->Headers->HasKey(keyRequested)) {
        String^ value = m_responseMessage->Content->Headers->Lookup(keyRequested);
        return StringUtil::toString(value);
    }

    return std::string();
}

void HttpResponse::read(ContentHandler contentHandler, ErrorHandler errorHandler)
{
    if (m_cts.get_token().is_canceled()) {
        TRACE_INFO("HttpResponse::read() %p Request is cancelled", this);
        return;
    }

    auto reader = std::make_shared<StreamReader>(contentHandler, m_cts.get_token());
    m_timer = std::make_shared<windows::HttpTimer>(getReadTimeout(), m_syncPoint);

    auto self = shared_from_this();

    m_timer->setTimeoutFunction([self]() {
        TRACE_ERROR("HttpResponse::read() %p Task cancelled due to timeout", self.get());
        self->cancel();
    });

    auto task = create_task(
        m_responseMessage->Content->ReadAsInputStreamAsync(),
        m_cts.get_token());

    // This is a value continuation task
    auto inputStreamTask = task.then(
        [timer = m_timer, reader, errorHandler, self](IInputStream^ stream) {
        TRACE_DEBUG("HttpResponse::read() %p StreamReader->read", self.get());
        try {
            auto readerTask = reader->read(stream, timer);
            readerTask.get(); // force throw of canceled task
            return readerTask;
        } catch (const task_canceled&) {
            if (timer->hasTimedOut()) {
                TRACE_ERROR("HttpResponse::read() %p Task cancelled due to timeout in task_canceled (StreamReader->read())", self.get());
                errorHandler(static_cast<int>(HttpStatusCode::RequestTimeout));
            } else {
                TRACE_DEBUG("HttpResponse::read() %p Task cancelled by user (StreamReader->read())", self.get());
            }

            cancel_current_task();
        }

    }, m_cts.get_token(), task_continuation_context::use_current());

    inputStreamTask.then(
        [timer = m_timer, errorHandler, self](concurrency::task<IBuffer^> readStreamTask) {
        try {
            timer->setTimeoutFunction(nullptr);
            readStreamTask.get();
        }
        catch (const task_canceled&) {
            try {
                if (timer->hasTimedOut()) {
                    TRACE_ERROR("HttpResponse::read() %p Task cancelled due to timeout in task_canceled", self.get());
                    errorHandler(static_cast<int>(HttpStatusCode::RequestTimeout));
                } else {
                    TRACE_DEBUG("HttpResponse::read() %p Task cancelled by user", self.get());
                }
            } catch (::Platform::Exception ^e) {
                TRACE_ERROR("HttpResponse::read() %p Caught exception: %s (Case 3)", self.get(), StringUtil::toString(e->Message).c_str());
                errorHandler(e->HResult);
            } catch (const std::exception& e) {
                TRACE_ERROR("HttpResponse::read() %p Caught exception: %s (Case 1)", self.get(), e.what());
                errorHandler(0);
            } catch (...) {
                TRACE_ERROR("HttpResponse::read() %p Caught unknown exception (Case 4)", self.get());
                errorHandler(0);
            }
        } catch (::Platform::Exception ^ e) {
            TRACE_ERROR("HttpResponse::read() %p Caught exception: %s (Case 6)", self.get(), StringUtil::toString(e->Message).c_str());
            errorHandler(e->HResult);
        } catch (const std::exception& e) {
            TRACE_ERROR("HttpResponse::read() %p Caught exception: %s (Case 2)", self.get(), e.what());
            errorHandler(0);
        } catch (...) {
            TRACE_ERROR("HttpResponse::read() %p Caught unknown exception (Case 5)", self.get());
            errorHandler(0);
        }
    }, m_cts.get_token(), task_continuation_context::use_current());
}

void HttpResponse::cancel()
{
    TRACE_DEBUG("HttpResponse::cancel() %p", this);
    if (!m_cts.get_token().is_canceled()) {
        TRACE_DEBUG("HttpResponse::cancel() %p cancelling token", this);
        m_cts.cancel();
    }
}

void HttpResponse::onCancel()
{
    TRACE_DEBUG("HttpResponse::onCancel() %p", this);
    if (m_timer) {
        m_timer->stop();
    }
}

HttpResponse::StreamReader::StreamReader(ContentHandler contentHandler, concurrency::cancellation_token token)
    : m_contentHandler(contentHandler)
    , m_token(token)
{
    m_buffer = ref new Buffer(HttpBufferSize);
}

task<IBuffer^> HttpResponse::StreamReader::read(IInputStream^ stream, std::shared_ptr<windows::HttpTimer> timer)
{
    timer->start();

    timer->syncPoint(6);

    // Pass shared pointer to self since .then() lambda capture runs in a different thread
    auto self = shared_from_this();
    timer->syncPoint(7);
    return create_task(
        stream->ReadAsync(m_buffer, m_buffer->Capacity, InputStreamOptions::Partial),
        m_token
    ).then(
        [self, stream, timer](task<IBuffer^> readTask) {
            timer->syncPoint(8);
            while (true) {
                timer->syncPoint(9);
                try {
                    timer->syncPoint(10);
                    IBuffer^ readBuffer = readTask.get();
                    timer->syncPoint(11);
                    if (readBuffer->Length == 0) {
                        timer->syncPoint(12);
                        timer->stop();

                        if (self->m_token.is_canceled()) {
                            TRACE_DEBUG("HttpResponse::StreamReader::read %p was cancelled (checked manually)", self.get());
                            cancel_current_task();
                        } else {
                            self->m_contentHandler(nullptr, 0, true);
                            return readTask;
                        }
                    }

                    timer->syncPoint(13);

                    auto dataReader = Windows::Storage::Streams::DataReader::FromBuffer(readBuffer);
                    auto data = ref new ::Platform::Array<uint8>(readBuffer->Length);
                    dataReader->ReadBytes(data);

                    timer->syncPoint(14);
                    self->m_contentHandler(data->begin(), readBuffer->Length, false);
                    timer->syncPoint(15);
                    timer->restart();
                    readTask = create_task(stream->ReadAsync(self->m_buffer, self->m_buffer->Capacity, InputStreamOptions::Partial), self->m_token);
                    timer->syncPoint(16);
                } catch (const task_canceled&) {
                    TRACE_DEBUG("HttpResponse::StreamReader::read %p was cancelled", self.get());
                    cancel_current_task();
                } catch (...) {
                    TRACE_DEBUG("HttpResponse::StreamReader::read %p unknown exception", self.get());
                }
                timer->syncPoint(17);
            }
        },
        m_token,
        task_continuation_context::use_current()
    );
}

}
}
