#include "pch.h"
#include "HttpRequest.hpp"
#include "StringUtil.hpp"

using namespace Concurrency;
using namespace Windows::Foundation;
using namespace Windows::Web::Http;

namespace twitch {
namespace uwp {

HttpRequest::HttpRequest(Windows::Web::Http::HttpClient ^ client, const std::string& url, twitch::HttpMethod method)
    : twitch::HttpRequest(url, method)
    , m_client(client)
    , m_syncPoint(-1)
{
    m_cancelCallback = m_cts.get_token().register_callback(std::bind(&HttpRequest::onCancel, this));

#if HTTPTIMER_SIMULATE_TIMEOUT
    m_syncPoint = std::rand() % MaxSyncPoints;
    TRACE_DEBUG("HttpRequest %p will sync at point %d", this, m_syncPoint);
#endif
}

HttpRequest::~HttpRequest()
{
    m_cts.get_token().deregister_callback(m_cancelCallback);

    if (m_timer) {
        m_timer->setTimeoutFunction(nullptr);
        m_timer->stop();
    }
}

void HttpRequest::send(HttpClient::ResponseHandler onResponse, HttpClient::ErrorHandler onError)
{
    TRACE_DEBUG("HttpRequest::send() %p Requesting URL: %s", this, getUrl().c_str());

    if (m_cts.get_token().is_canceled()) {
        TRACE_INFO("HttpRequest::send() %p Request is cancelled", this);
        return;
    }

    auto requestMessage = createMessage();

    m_timer = std::make_shared<windows::HttpTimer>(getTimeout(), m_syncPoint);

    m_timer->setTimeoutFunction([weakSelf = std::weak_ptr<HttpRequest>(shared_from_this())]() {
        if (auto self = weakSelf.lock()) {
            TRACE_ERROR("HttpRequest::send() %p (timeOutFunction) Task cancelled due to timeout", self.get());
            self->cancel();
        }
    });

    auto httpTask = create_task(
        m_client->SendRequestAsync(requestMessage, HttpCompletionOption::ResponseHeadersRead),
        m_cts.get_token());

    m_timer->start();

    m_timer->syncPoint(0);

    // This is a value continuation, if the previous task was cancelled, this will run in the cancelled state
    auto httpResponseTask = httpTask.then(
        [timer = m_timer](HttpResponseMessage^ responseMessage) {
        timer->stop();
        return responseMessage;
    }, m_cts.get_token(), task_continuation_context::use_current());

    m_timer->syncPoint(1);

    // So we can just try to run a .get() and it will trigger the cancel if that was the case, or an exception
    // if any network issue occurs. See HttpClient::SendRequestAsync documentation.
    try {
        m_timer->syncPoint(2);
        httpResponseTask.get();
        m_timer->syncPoint(3);
        m_timer->stop();
    } catch (const task_canceled&) {
        if (m_timer->hasTimedOut()) {
            TRACE_ERROR("HttpRequest::send() %p Task cancelled due to timeout before headers were read", this);
            onError(static_cast<int>(HttpStatusCode::RequestTimeout));
        } else {
            TRACE_DEBUG("HttpRequest::send() %p Task cancelled by user", this);
        }
        return;
    }
    catch (::Platform::Exception ^ e) {
        auto status = Windows::Web::WebError::GetStatus(e->HResult);
        auto intStatus = static_cast<int>(status);
        if (intStatus >= HttpStatus::HttpStatusContinue) {
            HttpResponseMessage^ responseMessage = ref new HttpResponseMessage(static_cast<HttpStatusCode>(status));
            TRACE_DEBUG("HttpRequest::send() Response is HttpStatusContinue (100)");
            auto response = std::make_shared<HttpResponse>(responseMessage, m_syncPoint);
            onResponse(response);
        }
        else {
            TRACE_ERROR("HttpRequest::send() Task got Exception %s", StringUtil::toString(e->Message).c_str());
            onError(intStatus);
        }
        return;
    }

    if (m_cts.get_token().is_canceled()) {
        if (m_timer->hasTimedOut()) {
            TRACE_ERROR("HttpRequest::read() %p was cancelled at sync point (3)");
            onError(static_cast<int>(HttpStatusCode::RequestTimeout));
        } else {
            TRACE_DEBUG("HttpRequest::send() %p Task cancelled by user", this);
        }
        return;
    }

    httpResponseTask.then(
        [timer = m_timer, onResponse, onError, syncPoint = m_syncPoint](task<HttpResponseMessage^> task) {
        HttpResponseMessage^ responseMessage;
        try {
            responseMessage = task.get();

            if (responseMessage && !timer->hasTimedOut()) {
                auto response = std::make_shared<HttpResponse>(responseMessage, syncPoint);
                onResponse(response);
            } else {
                TRACE_ERROR("HttpRequest::send() Task cancelled due to timeout (syncPoint 5)");
                onError(static_cast<int>(HttpStatusCode::RequestTimeout));
            }
        } catch (const task_canceled&) {
            if (timer->hasTimedOut()) {
                TRACE_ERROR("HttpRequest::send() Task cancelled due to timeout");
                onError(static_cast<int>(HttpStatusCode::RequestTimeout));
            } else {
                TRACE_DEBUG("HttpRequest::send() Task cancelled by user");
            }
        } catch (::Platform::Exception ^ e) {
            auto status = Windows::Web::WebError::GetStatus(e->HResult);
            auto intStatus = static_cast<int>(status);
            if (intStatus >= HttpStatus::HttpStatusContinue) {
                responseMessage = ref new HttpResponseMessage(static_cast<HttpStatusCode>(status));
                TRACE_DEBUG("HttpRequest::send() Response is HttpStatusContinue (100)");
                auto response = std::make_shared<HttpResponse>(responseMessage, syncPoint);
                onResponse(response);
            } else {
                TRACE_ERROR("HttpRequest::send() Task got Exception %s", StringUtil::toString(e->Message).c_str());
                onError(intStatus);
            }
        }
    }, m_cts.get_token(), task_continuation_context::use_current());
}

void HttpRequest::setHeader(const std::string& key, const std::string& value)
{
    m_headers[key] = value;
}
void HttpRequest::setContent(const std::vector<uint8_t>& content)
{
    (void)content;
    // TODO support request content
}

void HttpRequest::cancel()
{
    TRACE_DEBUG("HttpRequest::cancel()");
    if (!m_cts.get_token().is_canceled()) {
        m_cts.cancel();
    }
}

void HttpRequest::onCancel()
{
    TRACE_DEBUG("HttpRequest::onCancel()");
    if (m_timer) {
        m_timer->stop();
        m_timer->setTimeoutFunction(nullptr);
    }
}

HttpRequestMessage^ HttpRequest::createMessage()
{
    auto method = ref new Windows::Web::Http::HttpMethod(StringUtil::toString(getMethodString()));
    auto uri = ref new Uri(StringUtil::toString(getUrl()));

    HttpRequestMessage^ message = ref new Windows::Web::Http::HttpRequestMessage(method, uri);
    for (const auto& itr : m_headers) {
        auto key = StringUtil::toString(itr.first);
        auto value = StringUtil::toString(itr.second);
        message->Headers->Append(key, value);
    }

    return message;
}

}
}
