#include "pch.h"
#include "XAudioMonitor.hpp"
#include "debug/trace.hpp"

using namespace twitch;
using namespace twitch::uwp;

using namespace Microsoft;
using namespace Microsoft::WRL;

XAudioMonitor::Timer::Timer(std::chrono::seconds period)
    : m_period(period)
{
}

void XAudioMonitor::Timer::update(MediaTime time)
{
    if (m_start == MediaTime::zero()) {
        TRACE_DEBUG("XAudioMonitor::Timer::update(): First call to update. Setting start time=%ld microseconds", time.microseconds().count());
        m_start = time;
        assert(m_numTicks == 0);
        m_numTicks = 0;
        return;
    }

    auto elapsed = time - m_start;
    auto numTicks = static_cast<size_t>(elapsed.milliseconds() / m_period);

    if (numTicks > m_numTicks) {
        // Check if we missed a tick/period.  For now, we'll ignore and fire just one event
        if (numTicks > m_numTicks + 1) {
            TRACE_DEBUG("XAudioMonitor::Timer::update(): numTicks (%ld) > m_numTicks (%ld) + 1", numTicks, m_numTicks);
        }

        if (onTick) {
            onTick();
        }
        m_numTicks = numTicks;
    }
}

XAudioMonitor::XAudioMonitor(Microsoft::WRL::ComPtr<IXAudio2>& xAudio2)
    : m_xAudio2(xAudio2)
{
    m_timer.onTick = std::bind(&XAudioMonitor::print, this);
}

void XAudioMonitor::print()
{
    XAUDIO2_PERFORMANCE_DATA perfData;
    m_xAudio2->GetPerformanceData(&perfData);

    TRACE_INFO("XAudio2 - Perf: Mem: %u bytes, Sample Latency: %u, Glitches: %u, Active Source Voice: %u, Total Source Voice: %u",
        perfData.MemoryUsageInBytes,
        perfData.CurrentLatencyInSamples,
        perfData.GlitchesSinceEngineStarted,
        perfData.ActiveSourceVoiceCount,
        perfData.TotalSourceVoiceCount);
}

void XAudioMonitor::update(const twitch::MediaSample& input)
{
    m_timer.update(input.decodeTime);
}
