$(function() {
    "use strict";

    // The max bitrate of variant that plays well
    var MAX_BITRATE = 5e6;
    var MAX_FRAMERATE = 55;

    window.castReceiverManager = null;
    window.mediaManager = null;
    window.messageBus = null;
    window.mediaElement = null;
    window.connectedCastSenders = [];

    window.castReceiverManager = cast.receiver.CastReceiverManager.getInstance();

    // Initialize the receiver SDK before starting the app-specific logic
    window.mediaPlayer = prepareMediaPlayer();
    var videoContainer = document.getElementById('video-container');
    videoContainer.appendChild(window.mediaPlayer.getHTMLVideoElement());

    // Twitch specific
    window.appVersion = "Chromecast 2.0";

    window.idleTimeout = null; // timeout for closing receiver if senders idle
    window.splashTimeout = null; // timeout that transitions from splash screen to idle
    window.metadataTimeout = null;
    window.updateStreamInfoInterval = null;
    window.channelName = null;
    window.vodId = null;
    window.broadcastId = null;
    window.playSessionId = null;
    window.clipPlayback = false;
    window.vodViews = 0;
    window.gameName = null;
    window.analyticsBlob = null;
    window.currentlyPlaying = false;

    function setHudMessage(elementId, message) {
        document.getElementById(elementId).innerHTML = '' + JSON.stringify(message);
    }

    setHudMessage('applicationState','Loaded. Starting up.');

    /**
     * Sets the log verbosity level.
     *
     * Debug logging (all messages).
     * DEBUG
     *
     * Verbose logging (sender messages).
     * VERBOSE
     *
     * Info logging (events, general logs).
     * INFO
     *
     * Error logging (errors).
     * ERROR
     *
     * No logging.
     * NONE
     **/
    cast.receiver.logger.setLevelValue(cast.receiver.LoggerLevel.DEBUG);

    /**
     * Called to process 'ready' event. Only called after calling castReceiverManager.start(config) and the
     * system becomes ready to start receiving messages.
     *
     * @param {cast.receiver.CastReceiverManager.Event} event - can be null
     *
     * There is no default handler
     */
    window.castReceiverManager.onReady = function(event) {
        console.log("### Cast Receiver Manager is READY: ", event);
        setHudMessage('castReceiverManagerMessage', 'READY: ' + JSON.stringify(event));
        setHudMessage('applicationState','Loaded. Started. Ready.');

        window.splashTimeout = setTimeout(function() {
            showIdleScreen();
        }, 2000);
    }

    /**
     * If provided, it processes the 'senderconnected' event.
     * Called to process the 'senderconnected' event.
     * @param {cast.receiver.CastReceiverManager.Event} event - can be null
     *
     * There is no default handler
     */
    window.castReceiverManager.onSenderConnected = function(event) {
        console.log("### Cast Receiver Manager - Sender Connected : ", event);
        setHudMessage('castReceiverManagerMessage', 'Sender Connected: ' + JSON.stringify(event));

        // TODO - add sender and grab CastChannel from CastMessageBus.getCastChannel(senderId)
        var senders = window.castReceiverManager.getSenders();
        window.senderChannels = [];
        for (var index = 0; index < senders.length; index++) {
            window.senderChannels.push(messageBus.getCastChannel(senders[index]));
        }
        setHudMessage('sessionCount', '' + senders.length);
    }

    /**
     * If provided, it processes the 'senderdisconnected' event.
     * Called to process the 'senderdisconnected' event.
     * @param {cast.receiver.CastReceiverManager.Event} event - can be null
     *
     * There is no default handler
     */
    window.castReceiverManager.onSenderDisconnected = function(event) {
        console.log("### Cast Receiver Manager - Sender Disconnected : " , event);
        setHudMessage('castReceiverManagerMessage', 'Sender Disconnected: ' + JSON.stringify(event));

        var senders = window.castReceiverManager.getSenders();
        setHudMessage('sessionCount', '' + senders.length);

        //If last sender explicity disconnects, turn off
        if (senders.length == 0) {
            if (event.reason == cast.receiver.system.DisconnectReason.REQUESTED_BY_SENDER) {
                window.close();
            } else if (!window.currentlyPlaying) {
                // if implicit disconnect, only start idle timeout if video is not already playing
                startIdleTimeout();
            }
        }
    }

    /**
     * If provided, it processes the 'systemvolumechanged' event.
     * Called to process the 'systemvolumechanged' event.
     * @param {cast.receiver.CastReceiverManager.Event} event - can be null
     *
     * There is no default handler
     */
    window.castReceiverManager.onSystemVolumeChanged = function(event) {
        console.log("### Cast Receiver Manager - System Volume Changed : ", event);
        setHudMessage('castReceiverManagerMessage', 'System Volume Changed: ' + JSON.stringify(event));

        // See cast.receiver.media.Volume
        console.log("### Volume: " + event.data['level'] + " is muted? " + event.data['muted']);
        setHudMessage('volumeMessage', 'Level: ' + event.data['level'] + ' -- muted? ' + event.data['muted']);
    }

    /**
     * Called to process the 'visibilitychanged' event.
     *
     * Fired when the visibility of the application has changed (for example
     * after a HDMI Input change or when the TV is turned off/on and the cast
     * device is externally powered). Note that this API has the same effect as
     * the webkitvisibilitychange event raised by your document, we provided it
     * as CastReceiverManager API for convenience and to avoid a dependency on a
     * webkit-prefixed event.
     *
     * @param {cast.receiver.CastReceiverManager.Event} event - can be null
     *
     * There is no default handler for this event type.
     */
    window.castReceiverManager.onVisibilityChanged = function(event) {
        console.log("### Cast Receiver Manager - Visibility Changed : ",  event);
        setHudMessage('castReceiverManagerMessage', 'Visibility Changed: ' + JSON.stringify(event));

        /** check if visible and pause media if not - add a timer to tear down after a period of time
           if visibilty does not change back **/
        if (event.isVisible) { // It is visible
            window.mediaPlayer.play(); // Resume media playback
            window.clearTimeout(window.timeout); // Turn off the timeout
            window.timeout = null;
        } else {
            window.mediaPlayer.pause(); // Pause playback
            window.timeout = window.setTimeout(function(){window.close();},600000); // 10 Minute timeout
        }
    }

    /**
     * Use the messageBus to listen for incoming messages on a virtual channel using a namespace string.
     * Also use messageBus to send messages back to a sender or broadcast a message to all senders.
     * You can check the cast.receiver.CastMessageBus.MessageType that a message bus processes though a call
     * to getMessageType. As well, you get the namespace of a message bus by calling getNamespace()
     */
    window.messageBus = window.castReceiverManager.getCastMessageBus('urn:x-cast:com.twitch.custom');
    /**
     * The namespace urn:x-cast:com.google.devrel.custom is used to identify the protocol of showing/hiding
     * the heads up display messages (The messages defined at the beginning of the html).
     *
     * The protocol consists of one string message: show
     * In the case of the message value not being show - the assumed value is hide.
     **/
    window.messageBus.onMessage = function(event) {
        console.log("### Message Bus - Media Message: ", event);
        setHudMessage('messageBusMessage', event);

        var jsonData = JSON.parse(event['data']);
        if ('config' in jsonData) {
            if (jsonData['config']['showOverlay']) {
                document.getElementById('messages').style.display = 'block';
            } else {
                document.getElementById('messages').style.display = 'none';
            }
        }

        if ('analytics' in jsonData) {
            setAnalyticsData(jsonData['analytics']);
        }

        if ('quality' in jsonData) {
            var group = jsonData.quality;
            if (group === 'auto') {
                window.mediaPlayer.setAutoSwitchQuality(true);
            } else {
                const quality = window.mediaPlayer.getQualities().find(q => q.group === group);
                if (typeof quality !== 'undefined') {
                    window.mediaPlayer.setQuality(quality);
                }
            }
        }
    }

    // This class is used to send/receive media messages/events using the media protocol/namesapce (urn:x-cast:com.google.cast.media).
    window.mediaManager = new cast.receiver.MediaManager(window.mediaPlayer, cast.receiver.media.Command.STREAM_VOLUME | cast.receiver.media.Command.STREAM_MUTE);

    /**
     * Called when the media ends.
     *
     * mediaManager.resetMediaElement(cast.receiver.media.IdleReason.FINISHED);
     **/
    window.mediaManager['onEndedOrig'] = window.mediaManager.onEnded;
    /**
     * Called when the media ends
     */
    window.mediaManager.onEnded = function() {
        console.log("### Media Manager - ENDED" );
        setHudMessage('mediaManagerMessage', 'ENDED');

        window.mediaManager['onEndedOrig']();
    }

    /**
     * Default implementation of onError.
     *
     * mediaManager.resetMediaElement(cast.receiver.media.IdleReason.ERROR)
     **/
    window.mediaManager['onErrorOrig'] = window.mediaManager.onError;
    /**
     * Called when there is an error not triggered by a LOAD request
     * @param obj
     */
    window.mediaManager.onError = function(obj) {
        try {
            console.log("### Media Manager - error: ", obj);
            setHudMessage('mediaManagerMessage', 'ERROR - ' + obj.type);
        } catch (e) {
            console.error('Media Manager Error handler error', e);
        }

        window.mediaManager['onErrorOrig'](obj);
    }

    /**
     * Processes the get status event.
     *
     * Sends a media status message to the requesting sender (event.data.requestId)
     **/
    window.mediaManager['onGetStatusOrig'] = window.mediaManager.onGetStatus;
    /**
     * Processes the get status event.
     * @param event
     */
    window.mediaManager.onGetStatus = function(event) {
        console.log("### Media Manager - GET STATUS: ", event);
        setHudMessage('mediaManagerMessage', 'GET STATUS ' + JSON.stringify(event));

        window.mediaManager['onGetStatusOrig'](event);
    }

    /**
     * Default implementation of onLoadMetadataError.
     *
     * mediaManager.resetMediaElement(cast.receiver.media.IdleReason.ERROR, false);
     * mediaManager.sendLoadError(cast.receiver.media.ErrorType.LOAD_FAILED);
     **/
    window.mediaManager['onLoadMetadataErrorOrig'] = window.mediaManager.onLoadMetadataError;
    /**
     * Called when load has had an error, overridden to handle application specific logic.
     * @param event
     */
    window.mediaManager.onLoadMetadataError = function(event) {
        console.log("### Media Manager - LOAD METADATA ERROR: ", event);
        setHudMessage('mediaManagerMessage', 'LOAD METADATA ERROR: ' + JSON.stringify(event));
        window.mediaManager['onLoadMetadataErrorOrig'](event);
    }

    /**
     * Default implementation of onMetadataLoaded
     *
     * Passed a cast.receiver.MediaManager.LoadInfo event object
     * Sets the mediaElement.currentTime = loadInfo.message.currentTime
     * Sends the new status after a LOAD message has been completed succesfully.
     * Note: Applications do not normally need to call this API.
     * When the application overrides onLoad, it may need to manually declare that
     * the LOAD request was sucessful. The default implementaion will send the new
     * status to the sender when the video/audio element raises the
     * 'loadedmetadata' event.
     * The default behavior may not be acceptable in a couple scenarios:
     *
     * 1) When the application does not want to declare LOAD succesful until for
     *    example 'canPlay' is raised (instead of 'loadedmetadata').
     * 2) When the application is not actually loading the media element (for
     *    example if LOAD is used to load an image).
     **/
    window.mediaManager['onLoadMetadataOrig'] = window.mediaManager.onLoadMetadataLoaded;
    /**
     * Called when load has completed, overridden to handle application specific logic.
     * @param event
     */
    window.mediaManager.onLoadMetadataLoaded = function(event) {
        console.log("### Media Manager - LOADED METADATA: ", event);
        setHudMessage('mediaManagerMessage', 'LOADED METADATA: ' + JSON.stringify(event));

        window.mediaManager['onLoadMetadataOrig'](event);
    }

    /**
     * Processes the pause event.
     *
     * mediaElement.pause();
     * Broadcast (without sending media information) to all senders that pause has happened.
     **/
    window.mediaManager['onPauseOrig'] = window.mediaManager.onPause;
    /**
     * Process pause event
     * @param event
     */
    window.mediaManager.onPause = function(event) {
        console.log("### Media Manager - PAUSE: ", event);
        setHudMessage('mediaManagerMessage', 'PAUSE: ' + JSON.stringify(event));
        window.mediaManager['onPauseOrig'](event);
    }

    /**
     * Default - Processes the play event.
     *
     * mediaElement.play();
     *
     **/
    window.mediaManager['onPlayOrig'] = window.mediaManager.onPlay;
    /**
     * Process play event
     * @param event
     */
    window.mediaManager.onPlay = function(event) {
        console.log("### Media Manager - PLAY: ", event);
        setHudMessage('mediaManagerMessage', 'PLAY: ' + JSON.stringify(event));

        $('#buffering_indicator').hide();
        window.mediaManager['onPlayOrig'](event);
    }

    /**
     * Default implementation of the seek event.
     * Sets the mediaElement.currentTime to event.data.currentTime.
     * If the event.data.resumeState is cast.receiver.media.SeekResumeState.PLAYBACK_START and the mediaElement is paused then
     * call mediaElement.play(). Otherwise if event.data.resumeState is cast.receiver.media.SeekResumeState.PLAYBACK_PAUSE and
     * the mediaElement is not paused, call mediaElement.pause().
     * Broadcast (without sending media information) to all senders that seek has happened.
     **/
    window.mediaManager['onSeekOrig'] = window.mediaManager.onSeek;
    /**
     * Process seek event
     * @param event
     */
    window.mediaManager.onSeek = function(event) {
        console.log("### Media Manager - SEEK: ", event);
        setHudMessage('mediaManagerMessage', 'SEEK: ' + JSON.stringify(event));

        window.mediaManager['onSeekOrig'](event);
    }

    /**
     * Default implementation of the set volume event.
     * Checks event.data.volume.level is defined and sets the mediaElement.volume to the value
     * Checks event.data.volume.muted is defined and sets the mediaElement.muted to the value
     * Broadcasts (without sending media information) to all senders that the volume has changed.
     **/
    window.mediaManager['onSetVolumeOrig'] = window.mediaManager.onSetVolume;
    /**
     * Process set volume event
     * @param event
     */
    window.mediaManager.onSetVolume = function(event) {
        console.log("### Media Manager - SET VOLUME: ", event);
        setHudMessage('mediaManagerMessage', 'SET VOLUME: ' + JSON.stringify(event));

        window.mediaManager['onSetVolumeOrig'](event);
    }

    /**
     * Processes the stop event.
     *
     * window.mediaManager.resetMediaElement(cast.receiver.media.IdleReason.CANCELLED, true, event.data.requestId);
     *
     * Resets Media Element to IDLE state. After this call the mediaElement
     * properties will change, paused will be true, currentTime will be zero and
     * the src attribute will be empty. This only needs to be manually called if the
     * developer wants to override the default behavior of onError, onStop or
     * onEnded, for example.
     **/
    window.mediaManager['onStopOrig'] = window.mediaManager.onStop;
    /**
     * Process stop event
     * @param event
     */
    window.mediaManager.onStop = function(event) {
        console.log("### Media Manager - STOP: ", event);
        setHudMessage('mediaManagerMessage', 'STOP: ' + JSON.stringify(event));

        window.mediaManager['onStopOrig'](event);
    }

    /**
     * Default implementation for the load event.
     *
     * Sets the mediaElement.autoplay to false.
     * Checks that data.media and data.media.contentId are valid then sets the mediaElement.src to the
     * data.media.contentId.
     *
     * Checks the data.autoplay value:
     *   - if undefined sets mediaElement.autoplay = true
     *   - if has value then sets mediaElement.autoplay to that value
     **/
    window.mediaManager['onLoadOrig'] = window.mediaManager.onLoad;
    /**
     * Processes the load event.
     * @param event
     */
    window.mediaManager.onLoad = function(event) {
        console.log("### Media Manager - LOAD: ", event);
        setHudMessage('mediaManagerMessage', 'LOAD ' + JSON.stringify(event));

        window.clearInterval(window.updateStreamInfoInterval);
        window.updateStreamInfoInterval = null;

        window.gameName = null;
        window.vodId = null;
        window.channelName = null;
        window.broadcastId = null;
        window.playSessionId = null;
        window.vodViews = 0;
        window.clipPlayback = false;
        window.currentlyPlaying = false;

        if (event.data['media'] && event.data['media']['contentId']) {
            var url = event.data['media']['contentId'];
            var mp4String = '.mp4';
            window.clipPlayback = url.indexOf(mp4String) === (url.length - mp4String.length);

            // First determine if for a vod
            $('#view_separator').hide();
            $('#vod_progress').hide();
            if (event.data['media']['customData']) {
                if (event.data['media']['customData']['vod_id']) {
                    window.vodId = event.data['media']['customData']['vod_id'];
                }

                if (event.data['media']['customData']['views']) {
                    window.vodViews = event.data['media']['customData']['views'];
                }
            }

            if (window.vodId || window.clipPlayback) {
                // clear out vod progress text if it exists
                $('#current_vod_progress').text("");
                $('#remaining_vod_progress').text("");
                $('.progress_complete').width("0%");

                $('#vod_progress').show();
            }

            setMetadata(event.data['media']['metadata']);

            if (event.data['media']['customData']) {
                if (event.data['media']['customData']['channel']) {
                    window.channelName = event.data['media']['customData']['channel'];
                    if (!window.vodId) {
                        updateStreamInfo();
                    }
                }

                if (event.data['media']['customData']['analytics']) {
                    // NOTE: possible that this information has not arrived yet and will come through on the com.twitch.custom channel
                    setAnalyticsData(event.data['media']['customData']['analytics']);
                }
            }

            var urlObj = new URL(url);
            if (urlObj.searchParams.has('cdm')) {
                urlObj.searchParams.delete('cdm');
                event.data['media']['contentId'] = urlObj.toString();
            }

            window.mediaManager.setMediaInformation(event.data['media'], false);
            window.mediaManager['onLoadOrig'](event);
        }
    };

    window.mediaManager.customizedStatusCallback = function(status) {
        if (!window.vodId && !window.clipPlayback && window.currentlyPlaying) { // it is a live stream
            status.currentTime = window.mediaPlayer.getStartOffset();
        }
        if (status.media && window.currentlyPlaying) {
            status.media.duration = window.mediaPlayer.getDurationSec();
            status.playerState = window.currentlyPlaying ? 'PLAYING' : 'BUFFERING';
            if (window.currentlyPlaying) {
                status.customData = {
                    playerBackendType: 'chromecast-mediaplayer',
                    channelName: window.channelName || '',
                    offset: window.mediaPlayer.getStartOffset(),
                    broadcastId: window.broadcastId,
                    vodId: window.vodId || '',
                    playSessionId: window.playSessionId
                }
            }
        }
        return status;
    };

    /**
     * Application config
     **/
    var appConfig = new cast.receiver.CastReceiverManager.Config();

    /**
     * Text that represents the application status. It should meet
     * internationalization rules as may be displayed by the sender application.
     * @type {string|undefined}
     **/
    appConfig.statusText = 'Ready to play';

    /**
     * Maximum time in seconds before closing an idle
     * sender connection. Setting this value enables a heartbeat message to keep
     * the connection alive. Used to detect unresponsive senders faster than
     * typical TCP timeouts. The minimum value is 5 seconds, there is no upper
     * bound enforced but practically it's minutes before platform TCP timeouts
     * come into play. Default value is 10 seconds.
     * @type {number|undefined}
     **/
    //appConfig.maxInactivity = 6000; // 10 minutes for testing, use default 10sec in prod by not setting this value

    /**
     * Initializes the system manager. The application should call this method when
     * it is ready to start receiving messages, typically after registering
     * to listen for the events it is interested on.
     */
    window.castReceiverManager.start(appConfig);

    function prepareMediaPlayer() {
        var mediaPlayer = new TwitchChromecastMediaPlayer({
            logLevel: 'debug',
            keySystem: '', // Disable DRM streams
            isQualitySupported: function(receiverManager, quality) {
                // In case of variant manifest urls quality codec strings is empty and we should support those
                return quality.codecs === '' ||
                       (
                           // MediaSource is type supported
                           MediaSource.isTypeSupported('video/mp4;codecs="' + quality.codecs + '"') &&
                           // AND receiverManager can display type
                           receiverManager.canDisplayType('video/mp4', quality.codecs, quality.width, quality.height, quality.framerate) &&
                           (
                                // chromecast is ultra or higher
                                receiverManager.canDisplayType('video/mp4', "avc1.4D402A,mp4a.40.2") ||
                                // OR bitrate is less than MAX_BITRATE
                                quality.bitrate < MAX_BITRATE ||
                                // OR framerate is less than MAX_FRAMERATE
                                Math.ceil(quality.framerate) <= MAX_FRAMERATE
                            )
                        );
            }.bind(undefined, window.castReceiverManager)
        });
        mediaPlayer.addEventListener(TwitchChromecastMediaPlayer.PlayerState.PLAYING, function(e){
            console.log("######### MEDIA PLAYER PLAYING");
            setHudMessage('mediaPlayerState', TwitchChromecastMediaPlayer.PlayerState.PLAYING);
            $('#buffering_indicator').hide();

            window.currentlyPlaying = true;
            showPlayerScreen();
        });

        mediaPlayer.addEventListener(TwitchChromecastMediaPlayer.PlayerState.IDLE, function(e){
            console.log("######### MEDIA PLAYER PAUSE");
            var senders = window.castReceiverManager.getSenders();
            senders.forEach(function(sender) {
                window.mediaManager.sendStatus(sender);
            });
            setHudMessage('mediaPlayerState', TwitchChromecastMediaPlayer.PlayerState.IDLE);
            $('#buffering_indicator').hide();
            window.currentlyPlaying = false;
            flashMetadata(true);
            startIdleTimeout();
        });

        mediaPlayer.addEventListener(TwitchChromecastMediaPlayer.PlayerState.BUFFERING, function(e){
            console.log("######### MEDIA PLAYER BUFFERING");
            setHudMessage('mediaPlayerState', TwitchChromecastMediaPlayer.PlayerState.BUFFERING);
            $('#buffering_indicator').show();

            var senders = window.castReceiverManager.getSenders();
            senders.forEach(function(sender) {
                window.mediaManager.sendStatus(sender, 0, true);
            });
        });

        mediaPlayer.addEventListener(TwitchChromecastMediaPlayer.PlayerState.READY, function(e){
            console.log("######### MEDIA PLAYER READY");
            setHudMessage('mediaPlayerState', TwitchChromecastMediaPlayer.PlayerState.READY);

            window.clearTimeout(window.splashTimeout);
            showPlayerLoadingScreen();

            if (!window.vodId) {
                window.updateStreamInfoInterval = setInterval(function() {
                    updateStreamInfo();
                }, 3000 * 60);
            }

            // send qualities
            var senders = window.castReceiverManager.getSenders();
            var payload = JSON.stringify({
                currentQuality: mediaPlayer.getQuality(),
                qualities: mediaPlayer.getQualities(),
                auto: mediaPlayer.getAutoSwitchQuality(),
                event: TwitchChromecastMediaPlayer.PlayerState.READY
            });

            senders.forEach(function(sender) {
                window.messageBus.send(sender, payload);
            });

            window.mediaManager.sendLoadComplete();
        });

        mediaPlayer.addEventListener(TwitchChromecastMediaPlayer.PlayerEvent.TIME_UPDATE, function(e){
            if (window.vodId || window.clipPlayback) {
                updateVodProgress(window.mediaPlayer.getCurrentTimeSec(), window.mediaPlayer.getDurationSec());
            }
            var senders = window.castReceiverManager.getSenders();
            senders.forEach(function(sender) {
                window.mediaManager.sendStatus(sender, 0, true);
            });
        });

        mediaPlayer.addEventListener(TwitchChromecastMediaPlayer.PlayerEvent.DURATION_CHANGED, function(e){
            $('#view_separator').show();
            var duration = window.mediaPlayer.getDurationSec();
            var formattedDuration = convertSecondsToHMS(duration);
            $('.vod_duration').text(formattedDuration);

            var senders = window.castReceiverManager.getSenders();
            senders.forEach(function(sender) {
                window.mediaManager.sendStatus(sender, 0, true);
            });
        });

        mediaPlayer.addEventListener(TwitchChromecastMediaPlayer.PlayerState.ENDED, function(e){
            console.log("######### MEDIA PLAYER ENDED");
            setHudMessage('mediaPlayerState', TwitchChromecastMediaPlayer.PlayerState.ENDED);

            var senders = window.castReceiverManager.getSenders();
            senders.forEach(function(sender) {
                window.mediaManager.sendStatus(sender, 0, true);
            });

            window.currentlyPlaying = false;
            showIdleScreen();
            $('.view_count').text("");
            window.clearInterval(window.updateStreamInfoInterval);
            window.mediaManager.setIdleReason(cast.receiver.media.IdleReason.FINISHED);
        });


        mediaPlayer.addEventListener(TwitchChromecastMediaPlayer.PlayerEvent.REBUFFERING, function(e){
            var senders = window.castReceiverManager.getSenders();
            senders.forEach(function(sender) {
                window.mediaManager.sendStatus(sender, 0, true);
            });
        });

        mediaPlayer.addEventListener(TwitchChromecastMediaPlayer.PlayerEvent.ERROR, function(e){
            console.log("######### MEDIA PLAYER ERROR ", e);
            var type = e.type;

            switch (type) {
                case TwitchChromecastMediaPlayer.ErrorType.NOT_AVAILABLE:
                case TwitchChromecastMediaPlayer.ErrorType.AUTHORIZATION:
                    window.mediaManager.sendLoadError(cast.receiver.media.ErrorType.ERROR, {
                        source: e.source,
                        type: e.type,
                        message: e.message
                    });
                    break;
                default:
                    window.castReceiverManager.getSenders().forEach(function(sender) {
                        window.mediaManager.sendError(sender, 0, true, cast.receiver.media.ErrorType.ERROR, null, {
                            source: e.source,
                            type: e.type,
                            message: e.message
                        });
                    });
                    break;
            }

            var displayMessage = 'Sorry, something went wrong. Please try casting again';
            switch (type) {
                case TwitchChromecastMediaPlayer.ErrorType.NOT_AVAILABLE:
                    displayMessage = 'Stream is offline or currently not reachable';
                    break;

                case TwitchChromecastMediaPlayer.ErrorType.AUTHORIZATION:
                    displayMessage = 'Please restart the stream on your app and try again';
                    break;

                default:
                    break;
            }

            showError(displayMessage, '[type: ' + e.type + ' source: ' + e.source + ' code:' + e.code + ']');
            console.warn(displayMessage, '[type: ' + e.type + ' source: ' + e.source + ' code:' + e.code + ']')

            setHudMessage('mediaPlayerState', TwitchChromecastMediaPlayer.PlayerEvent.ERROR);
            window.currentlyPlaying = false;
            window.clearInterval(window.updateStreamInfoInterval);
        });

        mediaPlayer.addEventListener(TwitchChromecastMediaPlayer.PlayerEvent.TRACKING, function(event){
            if (event.name === 'video-play') {
                window.playSessionId = event.properties['play_session_id'];
                window.broadcastId = event.properties['broadcast_id'];
            }

            var eventName = event.name;
            if (eventName === 'minute-watched' && window.clipPlayback) {
                eventName = 'clips_minute_watched';
            }
            var eventProperties = Object.assign({}, event.properties, createGlobalMixpanelProperties());
            spade.track(eventName, eventProperties);
            mixpanel.track(eventName, eventProperties);
        });

        mediaPlayer.addEventListener(TwitchChromecastMediaPlayer.PlayerEvent.QUALITY_CHANGED, function(e){
            console.log("######### MEDIA PLAYER QUALITY CHANGE", e);

            // send qualities
            var senders = window.castReceiverManager.getSenders();
            var payload = JSON.stringify({
                currentQuality: mediaPlayer.getQuality(),
                qualities: mediaPlayer.getQualities(),
                auto: mediaPlayer.getAutoSwitchQuality(),
                event: TwitchChromecastMediaPlayer.PlayerEvent.QUALITY_CHANGED
            });

            senders.forEach(function(sender) {
                window.messageBus.send(sender, payload);
            });
        });

        return mediaPlayer;
    }

    /****************************
     * TWITCH UI LOGIC
     ****************************/

    function startIdleTimeout() {
        if (window.idleTimeout == null) {
            window.idleTimeout = setTimeout(function() {
                window.close();
            }, 5 * 1000 * 60)
        }
    }

    function showIdleScreen() {
        $('#splash_wrapper').hide();
        $('#idle_wrapper').show();
        $('#error_screen').hide();
        $('#playlist_notification').hide();
        $('#live_notification').hide();
        $('#paused_notification').hide();
        window.currentlyPlaying = false;
        startIdleTimeout();
    }

    function showPlayerLoadingScreen() {
        $('#splash_wrapper').hide();
        $('#idle_wrapper').hide();
        $('#error_screen').hide();
        $('#playlist_notification').hide();
        $('#live_notification').hide();
        $('#paused_notification').hide();

        $('#player_loading').show();
        $('#metadata_shade').show();
        $('#metadata').show();
        $('#buffering_indicator').show();

        clearTimeout(window.idleTimeout);
        window.idleTimeout = null;
    }

    function showPlayerScreen() {
        $('#splash_wrapper').hide();
        $('#idle_wrapper').hide();
        $('#error_screen').hide();

        $('#player_loading').hide();
        $('#buffering_indicator').hide();
        flashMetadata(false);

        clearTimeout(window.idleTimeout);
        window.idleTimeout = null;
    }

    function flashMetadata(forPause) {
        $('#metadata').show();
        $('#metadata_shade').show();

        if (forPause) {
            $('#playlist_notification').hide();
            $('#live_notification').hide();
            $('#paused_notification').show();
        } else {
            $('#paused_notification').hide();

            if (!window.vodId && !window.clipPlayback) {
                $('#live_notification').show();
                $('#playlist_notification').hide();
            }
        }

        clearTimeout(window.metadataTimeout);
        window.metadataTimeout = null;
        if (!forPause) {
            window.metadataTimeout = setTimeout(function() {
                $('#metadata').fadeOut();
                $('#metadata_shade').fadeOut();
                $('#live_notification').fadeOut();
                $('#paused_notification').fadeOut();
                $('#playlist_notification').fadeOut();
            }, 8000);
        }
    };

    function showError(errorMessage, details) {
        $('#splash_wrapper').hide();
        $('#idle_wrapper').hide();
        $('#live_notification').hide();
        $('#playlist_notification').hide();
        $('#paused_notification').hide();

        $('#error_screen').show();
        $('#error_message').text(errorMessage || 'Unknown reason');
        $('#error_details').text(details || 'Unknown details');
        startIdleTimeout();
    }


    function setMetadata(metadata) {
        $('.stream_name').text(metadata['title']);

        if (window.vodId) {
            // vod metadata
            $('#live_subtext').hide();
            $('#playlist_subtext').hide();
            $('#vod_subtext').show();
            $('.vod_views').text(window.vodViews);
            $('.vod_duration').text("");

        } else {
            // live metadata
            if (metadata['subtitle']) {
                window.gameName = metadata['subtitle'];
                $('#live_subtext').show();
                $('#playlist_subtext').hide();
                $('#vod_subtext').hide();
                $('.game_name').text(window.gameName);
            }
        }

        if (metadata['images'] && metadata['images'].length && metadata['images'][0]['url']) {
            $('#icon_wrapper').css('background-image', 'url(' + metadata['images'][0]['url'] + ')' );
        } else {
            $('#icon_wrapper').css('background-image', 'none');
        }
    }

    function jitterIdleScreen() {
        var $idle_wrapper = $('#idle_wrapper');
        var $idle_content = $('#idle_content');

        var positions = ['43%', '45%', '47%', '49%'];
        setInterval(function() {
            if ($idle_wrapper.is(':visible')) {
                $idle_content.css('top', positions[Math.floor(Math.random() * positions.length)]);
            }
        }, 15000);
    };
    jitterIdleScreen();

    function commaizeNumber(x) {
        return x.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ",");
    };

    function updateVodProgress(curTime, duration) {
        var $progressComplete = $('.progress_complete');
        if (duration !== Infinity) {
            var progress = curTime / duration * 100;
            $progressComplete.width(progress + "%");

            var formattedCurrent = convertSecondsToHMS(curTime);
            $('#current_vod_progress').text(formattedCurrent);

            var formattedRemaining = convertSecondsToHMS(duration - curTime);
            $('#remaining_vod_progress').text("-" + formattedRemaining);
        }
    };

    function convertSecondsToHMS(seconds) {
        var s = parseInt(seconds % 60);
        var m = parseInt((seconds / 60) % 60);
        var h = parseInt((seconds / (60*60)) % 24);
        return ((h > 0 ? h + ":" : "") + (m > 0 ? (h > 0 && m < 10 ? "0" : "") + m + ":" : "0:") + (s < 10 ? "0" : "") + s);
    };

    window.streamInfoCallback = function(data) {
        if (data['stream']) {
            if (data['stream']['game']) {
                window.gameName = data['stream']['game'];
                $('.game_name').text(window.gameName);
            }

            if (data['stream']['viewers']) {
                $('.view_count').html(' <span class="emphasis">' +
                    commaizeNumber(data['stream']['viewers']) +
                    '</span> viewers');
            }
        } else {
            //stream is offline
            window.channelName = null;
        }
    };

    function updateStreamInfo() {
        if (window.channelName) {
            var url = 'https://api.twitch.tv/kraken/streams/' + window.channelName;

            $.ajax({
                dataType: 'jsonp',
                url: url,
                jsonpCallback: 'streamInfoCallback',
                cache: true,  // prevent appending random garbage
                headers: {
                    'Accept': 'application/vnd.twitchtv.v3+json',
                    'Client-ID': 'fb2u5g608t1hvxklzk18vfrnqjxcdxl'
                }
            });
        }
    };


    /***************
     *MIXPANEL
     ****************/

    function setAnalyticsData(analytics) {
        window.analyticsBlob = analytics || {};
    };

    function getDistinctId() {
        var currentId = localStorage.getItem('distinct_id') || '';
        if (currentId.length > 0) {
            return currentId;
        }

        var distinct_id= "";
        var possible = "abcdef0123456789";

        for (var i=0; i < 32; i++) {
            distinct_id += possible.charAt(Math.floor(Math.random() * possible.length));
        }

        localStorage.setItem('distinct_id', distinct_id);
        return distinct_id;
    };

    // TODO Find ways to create the constant set of properties
    function createGlobalMixpanelProperties() {
        var id = window.analyticsBlob['distinct_id'] || getDistinctId();
        var properties = {
            distinct_id: id,
            device_id: id,
            app_version: window.appVersion,
            channel: window.channelName,
            chromecast_sender: window.analyticsBlob['chromecast_sender'] || 'unknown',
            partner: window.analyticsBlob['partner'] || 'unknown',
            player: 'chromecast-mediaplayer',
            platform: 'chromecast',
            content_mode: window.vodId ? 'vod' : window.clipPlayback ? 'clips' : 'live'
        }

        if (window.gameName) { // not set for vods or offline playlist
            properties['game'] = window.gameName;
        }

        if (window.vodId) {
            properties['vod_id'] = window.vodId;

            if ('vod_type' in window.analyticsBlob) {
                properties['vod_type'] = window.analyticsBlob['vod_type'];
            }
        }

        // Some senders do not explicitly provide logged_in (android / ios)
        if ('logged_in' in window.analyticsBlob) {
            properties['logged_in'] = window.analyticsBlob['logged_in'];
        } else if ('login' in window.analyticsBlob) {
            properties['logged_in'] = true;
        } else {
            properties['logged_in'] = false;
        }

        // If logged out, explicitly dont send subscriber, turbo, and login
        if (!properties['logged_in']) {
            return properties;
        }

        if ('subscriber' in window.analyticsBlob) {
            properties['subscriber'] = window.analyticsBlob['subscriber']
        }

        if ('turbo' in window.analyticsBlob) {
            properties['turbo'] = window.analyticsBlob['turbo']
        }
        if ('login' in window.analyticsBlob) {
            properties['login'] = window.analyticsBlob['login']
        }

        return properties;
    }
});
