#include "conversion.hpp"
#include <codecvt>
#include <locale>

using namespace json11;

namespace conversion {

// Javascript uses utf16 So we must convert
std::wstring utf8ToUtf16(const std::string& str)
{
    std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>> converter;
    return converter.from_bytes(str);
}

// Convert json11 to empscriten 'val' object,
// which represents a native javascript value
emscripten::val jsonToVal(const Json& json)
{
    switch (json.type()) {
    case Json::NUL:
        return emscripten::val::null();

    case Json::NUMBER:
        return emscripten::val(json.number_value());

    case Json::BOOL:
        return emscripten::val(json.bool_value());

    case Json::STRING:
        return emscripten::val(utf8ToUtf16(json.string_value()));

    case Json::ARRAY: {
        auto array = emscripten::val::array();

        for (const auto& item : json.array_items()) {
            array.call<void>("push", jsonToVal(item));
        }

        return array;
    }

    case Json::OBJECT: {
        auto obj = emscripten::val::object();

        for (const auto& kv : json.object_items()) {
            obj.set(utf8ToUtf16(kv.first), jsonToVal(kv.second));
        }

        return obj;
    }
    }
}

Json valToJson(const emscripten::val& val)
{
    static const emscripten::val Array = emscripten::val::global("Array");
    static const emscripten::val Object = emscripten::val::global("Object");

    // 'null' and 'Array' must be checked first explicitly,
    // since they both have a typeof 'object'
    if (val.isNull()) {
        return Json(nullptr);
    }

    if (Array.call<bool>("isArray", val)) {
        Json::array array;
        int length = val["length"].as<int>();

        for (int i = 0; i < length; ++i) {
            array.push_back(valToJson(val[i]));
        }

        return array;
    }

    auto valType = val.typeOf().as<std::string>();

    if (valType == "number") {
        return Json(val.as<double>());
    }

    if (valType == "string") {
        return Json(val.as<std::string>());
    }

    if (valType == "boolean") {
        return Json(val.as<bool>());
    }

    if (valType == "object") {
        Json::object object;
        auto keys = Object.call<emscripten::val>("keys", val);
        int numKeys = keys["length"].as<int>();

        for (int i = 0; i < numKeys; ++i) {
            auto key = keys[i].as<std::string>();
            object[key] = valToJson(val[key]);
        }

        return object;
    }

    // use null for unsupported types ("symbol", "function", "undefined")
    return Json(nullptr);
}

} //namespace conversion
