export interface AccessTokenConfig {
    streamId: string;
    streamType: 'channel' | 'vod';
    clientId: string;
    oAuthToken?: string;
    playerType: string;
    playerBackend: string;
    platform: string;
}

export interface TokenResponse {
    token: string;
    sig: string;
}

export function getAccessToken({ streamId, streamType, clientId, oAuthToken, playerType, playerBackend, platform }: AccessTokenConfig): Promise<TokenResponse> {
    const params = [
        `client_id=${clientId}`,
        `player_type=${playerType}`,
        `player_backend=${playerBackend}`,
        `platform=${platform}`,
    ];

    if (oAuthToken) {
        params.push(`oauth_token=${oAuthToken}`);
    }

    const url = `https://api.twitch.tv/api/${streamType}s/${streamId}/access_token?${params.join('&')}`;

    return new Promise((resolve, reject) => {
        const xhr = new XMLHttpRequest();
        xhr.open('GET', url);
        xhr.onerror = reject;
        xhr.onload = () => {
            if (xhr.status >= 200 && xhr.status < 300) {
                try {
                    resolve(JSON.parse(xhr.responseText));
                } catch (e) {
                    reject(e);
                }
            } else {
                reject(new Error(xhr.responseText));
            }
        };
        xhr.send();
    });
}

export function createMasterPlaylistUrl({ streamId, streamType, playerType, playerBackend }: AccessTokenConfig, { token, sig }: TokenResponse): string {
    const params = [
        'playlist_include_framerate=true',
        'allow_source=true',
        `player_type=${playerType}`,
        `player_backend=${playerBackend}`,
    ];

    let streamPath: string;

    if (streamType === 'vod') {
        streamPath = 'vod';
        params.push(
            `nauth=${encodeURIComponent(token)}`,
            `nauthsig=${sig}`,
        );
    } else {
        streamPath = 'api/channel/hls';
        params.push(
            `token=${encodeURIComponent(token)}`,
            `sig=${sig}`,
        );
    }

    return `https://usher.ttvnw.net/${streamPath}/${streamId}.m3u8?${params.join('&')}`;
}
