/**
 * https://developers.google.com/cast/docs/reference/receiver/cast.receiver.media.Player
 * Chromecast Player interface
 */
'use strict';

var MediaPlayerPackage = require('./index');
var MediaPlayer = MediaPlayerPackage.MediaPlayer;
var PlayerState = MediaPlayerPackage.PlayerState;
var PlayerEvent = MediaPlayerPackage.PlayerEvent;
var ErrorType = MediaPlayerPackage.ErrorType;
var ErrorSource = MediaPlayerPackage.ErrorSource;

var CastPlayerState = {
    IDLE:       'IDLE',
    PLAYING:    'PLAYING',
    PAUSED:     'PAUSED',
    BUFFERING:  'BUFFERING'
}

module.exports = ChromecastMediaPlayer;

var useWasm = typeof WebAssembly === 'object' && typeof WebAssembly.instantiate === 'function';
var WORKER_URL = document.currentScript.src.replace(/[\w-\.]+\.js$/g, useWasm ? WASM_WORKER_URL : ASMJS_WORKER_URL);
var WASM_URL = document.currentScript.src.replace(/[\w-\.]+\.js$/g, WASM_BINARY_URL);
var worker = MediaPlayer.createWorker(WORKER_URL, WASM_URL);

function ChromecastMediaPlayer (config) {
    this._config = config;
    this._prepareMediaPlayer(config);
}

ChromecastMediaPlayer.prototype._prepareMediaPlayer = function (config) {
    // Init mediaplayer instance with worker inited using Blob URL
    this._mediaPlayer = new MediaPlayer(config, worker);

    // Init states and other listeners
    this._videoElement = this._mediaPlayer.getHTMLVideoElement();
    this._state = CastPlayerState.IDLE;
    this._allListeners = [];
    this._attachListeners();
};

ChromecastMediaPlayer.prototype._onVideoPlaying = function () {
    this._state = CastPlayerState.PLAYING;
};

ChromecastMediaPlayer.prototype._onVideoPaused = function () {
    this._state = CastPlayerState.PAUSED;
};

ChromecastMediaPlayer.prototype._onVideoBuffering = function () {
    this._state = CastPlayerState.BUFFERING;
};

ChromecastMediaPlayer.prototype._attachListeners = function () {
    // Attach core listener
    this._addListener(this._mediaPlayer, PlayerState.BUFFERING, this._onVideoBuffering.bind(this));
    this._addListener(this._mediaPlayer, PlayerEvent.REBUFFERING, this._onVideoBuffering.bind(this));
    this._addListener(this._mediaPlayer, PlayerState.PLAYING, this._onVideoPlaying.bind(this));
    this._addListener(this._mediaPlayer, PlayerState.IDLE, this._onVideoPaused.bind(this));
}

ChromecastMediaPlayer.prototype._addListener = function (target, eventName, listener) {
    target.addEventListener(eventName, listener);
    this._allListeners.push({
        target: target,
        eventName: eventName,
        listener: listener
    });
};

ChromecastMediaPlayer.prototype._removeListener = function (target, eventName) {
    this._allListeners.forEach(function (entry) {
        if (entry.target === target && entry.eventName === eventName) {
            entry.target.removeEventListener(entry.eventName, entry.listener);
        }
    });
};

ChromecastMediaPlayer.prototype._removeAllListeners = function () {
    this._allListeners.forEach(function (entry) {
        entry.target.removeEventListener(entry.eventName, entry.listener);
    });
    this._allListeners = [];
};

ChromecastMediaPlayer.prototype.getHTMLVideoElement = function () {
    return this._videoElement;
};

ChromecastMediaPlayer.prototype.editTracksInfo = function () {
    // TODO Figure out what edits can be made
};

ChromecastMediaPlayer.prototype.getStartOffset = function () {
    return this._mediaPlayer.getStartOffset();
};

ChromecastMediaPlayer.prototype.getCurrentTimeSec = function () {
    return this._mediaPlayer.getPosition();
};

ChromecastMediaPlayer.prototype.getDurationSec = function () {
    return this._mediaPlayer.getDuration();
};

ChromecastMediaPlayer.prototype.getState = function () {
    return this._state;
};

ChromecastMediaPlayer.prototype.getVolume = function () {
    // Expects cast.receiver.media.Volume
    // https://developers.google.com/cast/docs/reference/receiver/cast.receiver.media.Volume.html
    return {
        level: this._mediaPlayer.getVolume(),
        muted: this._mediaPlayer.isMuted()
    };
};

ChromecastMediaPlayer.prototype.load = function (contentId, autoPlay, currentTime) {
    if (!this._mediaPlayer && this._config) {
        this._prepareMediaPlayer(this._config);
    }

    this._state = CastPlayerState.BUFFERING;

    // Load the content
    this._mediaPlayer.load(contentId);

    if (autoPlay) {
        this._mediaPlayer.play();
    }

    if (currentTime > 0) {
        this._mediaPlayer.seekTo(currentTime);
    }
};

ChromecastMediaPlayer.prototype.pause = function () {
    this._mediaPlayer.pause();
};

ChromecastMediaPlayer.prototype.play = function () {
    this._mediaPlayer.play();
};

ChromecastMediaPlayer.prototype.registerEndedCallback = function (cb) {
    this._addListener(this._mediaPlayer, PlayerState.ENDED, cb);
};

ChromecastMediaPlayer.prototype.registerErrorCallback = function (cb) {
    this._addListener(this._mediaPlayer, PlayerEvent.ERROR, cb);
};

ChromecastMediaPlayer.prototype.registerLoadCallback = function (cb) {
    this._addListener(this._mediaPlayer, PlayerState.READY, cb);
};

ChromecastMediaPlayer.prototype.reset = function () {
    this._state = CastPlayerState.IDLE;
    this._mediaPlayer.pause();
};

ChromecastMediaPlayer.prototype.seek = function (time) {
    this._mediaPlayer.seekTo(time);
};

/**
 * Sets the volume on player
 * @param volume {cast.receiver.media.Volume} The volume object
 */
ChromecastMediaPlayer.prototype.setVolume = function (volume) {
    if (!isNaN(volume.level)) {
        this._mediaPlayer.setVolume(volume.level);
    } else {
        this._mediaPlayer.setMuted(volume.muted);
    }
};

ChromecastMediaPlayer.prototype.unregisterEndedCallback = function () {
    this._removeListener(this._mediaPlayer, PlayerState.ENDED);
};

ChromecastMediaPlayer.prototype.unregisterErrorCallback = function () {
    this._removeListener(this._mediaPlayer, PlayerEvent.ERROR);
};

ChromecastMediaPlayer.prototype.unregisterLoadCallback = function () {
    this._removeListener(this._mediaPlayer, PlayerState.READY);
};

//Additional methods
ChromecastMediaPlayer.prototype.getQualities = function() {
    return this._mediaPlayer ? this._mediaPlayer.getQualities() : [];
};

ChromecastMediaPlayer.prototype.getQuality = function() {
    return this._mediaPlayer ? this._mediaPlayer.getQuality() : {};
};

ChromecastMediaPlayer.prototype.setQuality = function(quality) {
    if (this._mediaPlayer) {
        this._mediaPlayer.setQuality(quality);
    } else {
        console.warn('No mediaplayer available, cannot set quality');
    }
};

ChromecastMediaPlayer.prototype.addEventListener = function(event, listener) {
    if (this._mediaPlayer) {
        this._mediaPlayer.addEventListener(event, listener);
    } else {
        console.warn('No mediaplayer available, cannot add listener');
    }
};

ChromecastMediaPlayer.prototype.setAutoSwitchQuality = function(value) {
    if (this._mediaPlayer) {
        this._mediaPlayer.setAutoSwitchQuality(value);
    } else {
        console.warn('No mediaplayer available, cannot set auto');
    }
};

ChromecastMediaPlayer.prototype.getAutoSwitchQuality = function() {
    if (this._mediaPlayer) {
        return this._mediaPlayer.getAutoSwitchQuality();
    } else {
        console.warn('No mediaplayer available, cannot get auto');
    }
};

ChromecastMediaPlayer.prototype.getManifestInfo = function() {
    if (this._mediaPlayer) {
        return this._mediaPlayer.getManifestInfo();
    } else {
        console.warn('No mediaplayer available, cannot get manifest info');
    }
};

ChromecastMediaPlayer.prototype.destroy = function() {
    if (this._mediaPlayer) {
        this._removeAllListeners();
        this._mediaPlayer.delete();
        this._mediaPlayer = null;
        this._videoElement = null;
    } else {
        console.warn('No mediaplayer available, cannot set auto');
    }
};

ChromecastMediaPlayer.PlayerEvent = PlayerEvent;
ChromecastMediaPlayer.PlayerState = PlayerState;
ChromecastMediaPlayer.ErrorType = ErrorType;
ChromecastMediaPlayer.ErrorSource = ErrorSource;
