import { browser } from './browser';
import { MediaSink, MediaSinkListener, TrackConfig } from './mediasink';

const CAST_SENDER_URL = 'https://www.gstatic.com/cv/js/sender/v1/cast_sender.js?loadCastFramework=1';
const RECEIVER_APP_ID = 'F993ACB3'; // TODO Update to production
const CHROMECAST_SENDER_SCRIPT_ID = 'pc-chromecast-sender';
const TWITCH_NAMESPACE = 'urn:x-cast:com.twitch.custom';
const GENERIC_ERROR = 1;

export class ChromecastMediaSink implements MediaSink {
    static canCast(): boolean {
        return browser.chrome;
    }

    static async lookForRemotePlaybackDevices(listener: MediaSinkListener) {
        try {
            const castContext = await ChromecastMediaSink.prepareCastContext();
            castContext.addEventListener(cast.framework.CastContextEventType.CAST_STATE_CHANGED,
                (castStateEventData) => {
                    switch (castStateEventData.castState) {
                    case cast.framework.CastState.NO_DEVICES_AVAILABLE:
                        listener.onRemoteDevice(false);
                        break;
                    case cast.framework.CastState.NOT_CONNECTED:
                        listener.onRemoteDevice(true);
                        break;
                    default:
                        break;
                    }
                },
            );

            castContext.setOptions({
                receiverApplicationId: RECEIVER_APP_ID,
                autoJoinPolicy: chrome.cast.AutoJoinPolicy.ORIGIN_SCOPED,
            });
        } catch (err) {
            listener.onRemoteDevice(false);
        }
    }

    static async prepareCastContext(): Promise<cast.framework.CastContext> {
        if (window.cast && window.cast.framework) { // Framework is ready
            return Promise.resolve(cast.framework.CastContext.getInstance());
        }

        return new Promise((resolve, reject) => {
            const existingElement = document.getElementById(CHROMECAST_SENDER_SCRIPT_ID);
            if (existingElement) {
                document.body.removeChild(existingElement);
            }

            global['__onGCastApiAvailable'] = (isAvailable: boolean, message: string) => {
                if (isAvailable) {
                    resolve(cast.framework.CastContext.getInstance());
                } else {
                    reject();
                }
            };

            const scriptElement = document.createElement('script');
            scriptElement.id = CHROMECAST_SENDER_SCRIPT_ID;
            scriptElement.onerror = () => {
                document.body.removeChild(scriptElement);
                global['__onGCastApiAvailable'] = () => {
                    // noop
                };
                reject();
            };
            scriptElement.async = true;
            scriptElement.src = CAST_SENDER_URL;
            document.body.appendChild(scriptElement);
        });
    }

    private remotePlayer: cast.framework.RemotePlayer;
    private remotePlayerController: cast.framework.RemotePlayerController;
    private listener: MediaSinkListener;
    private seekTime: number;

    constructor(listener: MediaSinkListener) {
        this.listener = listener;
        ChromecastMediaSink.prepareCastContext()
        .then(() => {
            this.remotePlayer = new cast.framework.RemotePlayer();
            this.remotePlayerController = new cast.framework.RemotePlayerController(this.remotePlayer);
        })
        .catch(() => {
            this.listener.onSessionError();
        });
    }

    async configure({ path }: TrackConfig) {
        try {
            const castContext = await ChromecastMediaSink.prepareCastContext();
            let session = castContext.getCurrentSession();

            if (!session) {
                await castContext.requestSession();
                session = castContext.getCurrentSession();
                this.setupRemotePlayerListeners(session, this.remotePlayerController);
            } else if (session.getSessionState() === cast.framework.SessionState.SESSION_RESUMED) {
                this.setupRemotePlayerListeners(session, this.remotePlayerController);
            }

            // mediametadata start
            const mediaInfo = new chrome.cast.media.MediaInfo(path, '');
            mediaInfo.streamType = chrome.cast.media.StreamType.BUFFERED;

            const mediaMetadata = new chrome.cast.media.GenericMediaMetadata();

            mediaInfo.metadata = mediaMetadata;

            mediaInfo.customData = {
                analytics: {
                    chromecast_sender: 'player-core',
                    platform: 'web',
                },
            };
            // mediametadata end

            const loadRequest = new chrome.cast.media.LoadRequest(mediaInfo);
            await session.loadMedia(loadRequest);
        } catch (errorCode) {
            return this.handleError(errorCode);
        }
    }

    async stopMedia(stopCasting: boolean = true) {
        const castContext = await ChromecastMediaSink.prepareCastContext();
        const session = castContext.getCurrentSession();
        if (session && session.getSessionState() !== cast.framework.SessionState.SESSION_RESUMED) {
            session.endSession(stopCasting);
        }
    }

    enqueue() {
        // no-op
    }
    endOfStream() {
        // no-op
    }
    setTimestampOffset() {
        // no-op
    }

    play() {
        if (this.remotePlayer && this.remotePlayer.isPaused) {
            this.remotePlayerController.playOrPause();
        }
    }

    pause() {
        if (this.remotePlayer && !this.remotePlayer.isPaused) {
            this.remotePlayerController.playOrPause();
        }
    }

    reinit() {
        // no-op
    }
    remove() {
        // no-op
    }

    seekTo(time: number) {
        if (this.remotePlayer) {
            if (this.remotePlayer.playerState !== chrome.cast.media.PlayerState.IDLE) {
                this.remotePlayer.currentTime = time;
                this.remotePlayerController.seek();
            } else {
                this.seekTime = time; // Will be set when a mediasession starts
            }
        }
    }

    setPlaybackRate(rate: number) {
        // TODO
    }

    addCue() {
        // no-op
    }

    bufferDuration() {
        return 0;
    }

    buffered() {
        return { start: 0, end: 0 };
    }

    decodedFrames() {
        return 0;
    }

    droppedFrames() {
        return 0;
    }

    framerate() {
        return 0;
    }

    getDisplayWidth(): number {
        return 0;
    }

    getDisplayHeight(): number {
        return 0;
    }

    getPlaybackRate(): number {
        return 0;
    }
    getCurrentTime(): number {
        if (this.remotePlayer) {
            return this.remotePlayer.currentTime;
        }
        return 0;
    }
    delete() {
        if (this.remotePlayer) {
            this.stopMedia();
        }
    }

    setMuted(muted: boolean) {
        if (this.remotePlayer && muted !== this.remotePlayer.isMuted) {
            this.remotePlayerController.muteOrUnmute();
        }
    }

    isMuted(): boolean {
        return this.remotePlayer ? this.remotePlayer.isMuted : false;
    }

    setVolume(volume: number) {
        if (this.remotePlayer) {
            this.remotePlayer.volumeLevel = volume;
            this.remotePlayerController.setVolumeLevel();
        }
    }

    getVolume(): number {
        return this.remotePlayer ? this.remotePlayer.volumeLevel : 0;
    }

    async getDevice(): Promise<string> {
        const castContext = await ChromecastMediaSink.prepareCastContext();
        const session = castContext.getCurrentSession();
        if (session) {
            return session.getCastDevice().friendlyName;
        }
    }

    private setupRemotePlayerListeners(session: cast.framework.CastSession, remotePlayerController: cast.framework.RemotePlayerController) {
        const timeChangedHandler = async () => {
            const media = session.getMediaSession();
            if (media) {
                this.listener.onSinkTimeUpdate();
            }
        };

        const playerStateChangedHandler = () => {
            switch (this.remotePlayer.playerState) {
            case chrome.cast.media.PlayerState.BUFFERING:
                this.listener.onSinkIdle();
                break;
            case chrome.cast.media.PlayerState.PLAYING:
                this.listener.onSinkPlaying(false);
                break;
            case chrome.cast.media.PlayerState.IDLE:
                const media = session.getMediaSession();
                if (!media || media.idleReason === chrome.cast.media.IdleReason.FINISHED) {
                    this.listener.onSinkEnded();
                }
                break;
            default:
                break;
            }
        };

        const cleanUpHandlers = () => {
            this.remotePlayerController.removeEventListener(cast.framework.RemotePlayerEventType.CURRENT_TIME_CHANGED, timeChangedHandler);
            this.remotePlayerController.removeEventListener(cast.framework.RemotePlayerEventType.PLAYER_STATE_CHANGED, playerStateChangedHandler);
            this.seekTime = 0;
            this.listener.onSessionStop();
        };

        session.addEventListener(
            cast.framework.SessionEventType.MEDIA_SESSION,
            () => {
                remotePlayerController.addEventListener(cast.framework.RemotePlayerEventType.CURRENT_TIME_CHANGED, timeChangedHandler);
                this.remotePlayerController.addEventListener(cast.framework.RemotePlayerEventType.PLAYER_STATE_CHANGED, playerStateChangedHandler);
                if (this.seekTime > 0) {
                    this.remotePlayer.currentTime = this.seekTime;
                    this.remotePlayerController.seek();
                }
                this.listener.onSessionStarted(session.getCastDevice().friendlyName);
            },
        );

        const sessionObj = session.getSessionObj();
        sessionObj.addUpdateListener(() => { // User stopped the casting
            const status = sessionObj.status;
            if (status === chrome.cast.SessionStatus.STOPPED) {
                cleanUpHandlers();
            }
        });

        sessionObj.addMediaListener(cleanUpHandlers); // Casted by a different device
    }

    private handleError(errorCode: chrome.cast.ErrorCode) {
        if (chrome.cast) {
            switch (errorCode) {
            case chrome.cast.ErrorCode.SESSION_ERROR:
                this.listener.onSessionError();
                break;
            case chrome.cast.ErrorCode.RECEIVER_UNAVAILABLE:
                this.listener.onRemoteDevice(false);
                break;
            case chrome.cast.ErrorCode.LOAD_MEDIA_FAILED:
                this.listener.onLoadMediaError();
                break;
            case chrome.cast.ErrorCode.CANCEL:
                this.listener.onUserCancel();
                break;
            default:
                this.listener.onSinkError({
                    value: GENERIC_ERROR,
                    code: 0,
                    message: 'Error requesting chromecast session',
                });
                break;
            }
        } else {
            this.listener.onSinkError({
                value: GENERIC_ERROR,
                code: 0,
                message: 'Error loading chromecast SDK',
            });
        }

    }
}
