jest.mock('./drm/utils', () => {
    return {
        ...jest.requireActual('./drm/utils'),

        httpRequest: jest.fn().mockImplementation((url) => {
            if (url.indexOf('vizima') > -1) {
                return Promise.resolve(MOCK_VIZIMA_RESPONSE);
            } else if (url.indexOf('licensekeyserver') > -1) {
                return Promise.resolve(MOCK_LICENSE_SERVER_RESPONSE);
            } else {
                return Promise.reject();
            }
        }),

        parsePSSHSupportFromInitData: jest.fn().mockImplementation(() => {
            return [KEY_SYSTEMS.WIDEVINE];
        }),
    };
});

import { DRMManager } from './drm-manager';
import { AUTH_XML_URL, ERRORS, KEY_SYSTEMS, KeySystem } from './drm/constants';
import { httpRequest } from './drm/utils';
import { MOCK_LICENSE_SERVER_RESPONSE, MOCK_VIZIMA_RESPONSE } from './mocks/test-constants';
import { getFakeEventTarget } from './test-utils/fake-event-listener';
import { getFakeMediaSinkListener } from './test-utils/fake-helper';
import { addMissingVideoElementProperties } from './test-utils/video-element-api';

function setupDRMManager() {
    const video = document.createElement('video');

    const listener = getFakeMediaSinkListener();

    const specimen = new DRMManager({ video, listener });

    return { specimen, video, listener };
}

// This helps wait for all promises to end
function flushPromises() {
    return new Promise((resolve) => setImmediate(resolve));
}

function setupNavigatorToSupportKeySystem(
    keySystemSpecimen: KeySystem,
    shouldGenerateRequest: boolean = true,
    shouldSessionUpdate: boolean = true,
    sessionKeyStatuses: string[] = [],
) {
    if (keySystemSpecimen === null) {
        navigator.requestMediaKeySystemAccess = jest.fn().mockReturnValue(Promise.reject());
        return;
    }
    const keySession = {
        ...getFakeEventTarget(),
        generateRequest: jest.fn().mockReturnValue(shouldGenerateRequest ? Promise.resolve() : Promise.reject()),
        update: jest.fn().mockReturnValue(shouldSessionUpdate ? Promise.resolve() : Promise.reject()),
        keyStatuses: sessionKeyStatuses,
        close: jest.fn().mockReturnValue(Promise.resolve()),
    };

    const mediaKeys = {
        createSession: jest.fn().mockReturnValue(keySession),
        setServerCertificate: jest.fn(),
    } as MediaKeys;

    const mediaKeySystemAccess = {
        keySystem: keySystemSpecimen.keySystem,
        createMediaKeys: jest.fn().mockReturnValue(Promise.resolve(mediaKeys)),
        getConfiguration: {} as unknown,
    } as MediaKeySystemAccess;

    navigator.requestMediaKeySystemAccess = jest.fn().mockReturnValue(Promise.resolve(mediaKeySystemAccess));

    return {
        mediaKeys,
        keySession,
    };
}

beforeAll(() => {
    addMissingVideoElementProperties();
});

afterAll(() => {
    jest.clearAllMocks();
});

describe('DRMManager public methods', () => {
    afterEach(() => {
        navigator.requestMediaKeySystemAccess = jest.fn();
    });
    test('Test if configure call makes the correct request for authxml', () => {
        const { specimen } = setupDRMManager();

        specimen.configure('https://usher.ttvnw.net/api/channel/hls/randomchannel.m3u8?allow_source=true&player_backend=mediaplayer&sig=sig1&token=%7B%22adblock%22');
        expect(httpRequest).toBeCalledWith(`${AUTH_XML_URL}randomchannel?token=${'%7B%22adblock%22'}&sig=${'sig1'}`, {
            method: 'GET',
            responseType: 'text',
        });
    });

    test('Test configure and encrypted event handler for widewine keysystem', async () => {
        const { video } = setupDRMManager();

        const { mediaKeys } = setupNavigatorToSupportKeySystem(KEY_SYSTEMS.WIDEVINE);

        const setMediaKeysSpy = jest.spyOn(video, 'setMediaKeys');
        video.dispatchEvent(new Event('encrypted'));

        await flushPromises();
        expect(setMediaKeysSpy).toBeCalledWith(mediaKeys);
    });

    test('Test KeySession update license on widewine keysystem', async () => {
        const { keySession } = setupNavigatorToSupportKeySystem(KEY_SYSTEMS.WIDEVINE);

        const { video, specimen } = setupDRMManager();
        specimen.configure('https://usher.ttvnw.net/api/channel/hls/randomchannel.m3u8?allow_source=true&player_backend=mediaplayer&sig=sig1&token=%7B%22adblock%22');
        video.dispatchEvent(new Event('encrypted'));
        await flushPromises();

        // The flushPromises above is important because the keysession object is created after setmediakeys
        const updateSpy = jest.spyOn(keySession, 'update');
        keySession.dispatchEvent(new Event('message'), {
            message: new ArrayBuffer(5),
        });
        await flushPromises();

        expect(updateSpy).toBeCalledWith(MOCK_LICENSE_SERVER_RESPONSE);
    });

    test('Test for expired sessions', async () => {
        const { keySession } = setupNavigatorToSupportKeySystem(KEY_SYSTEMS.WIDEVINE, true, true, ['expired']);

        const { video, specimen } = setupDRMManager();
        specimen.configure('https://usher.ttvnw.net/api/channel/hls/randomchannel.m3u8?allow_source=true&player_backend=mediaplayer&sig=sig1&token=%7B%22adblock%22');
        video.dispatchEvent(new Event('encrypted'));
        await flushPromises();

        const closeSpy = jest.spyOn(keySession, 'close');
        keySession.dispatchEvent(new Event('keystatuseschange'), {
            message: new ArrayBuffer(5),
        });
        await flushPromises();

        expect(closeSpy).toBeCalled();
    });

    describe('Test onerror callback', () => {
        test('Test error ERRORS.AUTH_XML_REQUEST', async () => {
            const { keySession } = setupNavigatorToSupportKeySystem(KEY_SYSTEMS.WIDEVINE);

            const { video, listener } = setupDRMManager();
            video.dispatchEvent(new Event('encrypted'));

            await flushPromises();
            // The flushPromises above is important because the keysession object is created after setmediakeys
            keySession.dispatchEvent(new Event('message'), {
                message: new ArrayBuffer(5),
            });
            await flushPromises();
            expect(listener.onSinkError).toBeCalledWith(Object.assign({ code: 0 }, ERRORS.AUTH_XML_REQUEST));
        });

        test('Test error ERRORS.NO_CDM_SUPPORT', async () => {
            setupNavigatorToSupportKeySystem(null);
            const { video, listener } = setupDRMManager();
            video.dispatchEvent(new Event('encrypted'));

            await flushPromises();
            expect(listener.onSinkError).toBeCalledWith(Object.assign({ code: 0 }, ERRORS.NO_CDM_SUPPORT));
        });

        test('Test error ERRORS.KEY_SESSION_CREATION', async () => {
            setupNavigatorToSupportKeySystem(KEY_SYSTEMS.WIDEVINE, false);

            const { video, specimen, listener } = setupDRMManager();
            specimen.configure('https://usher.ttvnw.net/api/channel/hls/randomchannel.m3u8?allow_source=true&player_backend=mediaplayer&sig=sig1&token=%7B%22adblock%22');
            video.dispatchEvent(new Event('encrypted'));
            await flushPromises();
            expect(listener.onSinkError).toBeCalledWith(Object.assign({ code: 0 }, ERRORS.KEY_SESSION_CREATION));
        });

        test('Test error ERRORS.KEY_SESSION_INTERNAL', async () => {
            const { keySession } = setupNavigatorToSupportKeySystem(KEY_SYSTEMS.WIDEVINE, true, true, ['internal-error']);

            const { video, specimen, listener } = setupDRMManager();

            specimen.configure('https://usher.ttvnw.net/api/channel/hls/randomchannel.m3u8?allow_source=true&player_backend=mediaplayer&sig=sig1&token=%7B%22adblock%22');
            video.dispatchEvent(new Event('encrypted'));
            await flushPromises();

            keySession.dispatchEvent(new Event('keystatuseschange'), {
                message: new ArrayBuffer(5),
            });
            await flushPromises();

            expect(listener.onSinkError).toBeCalledWith(Object.assign({ code: 0 }, ERRORS.KEY_SESSION_INTERNAL));
        });

        test('Test error ERRORS.SESSION_UPDATE', async () => {
            const { keySession } = setupNavigatorToSupportKeySystem(KEY_SYSTEMS.WIDEVINE);

            const { video, specimen, listener } = setupDRMManager();
            specimen.configure('https://usher.ttvnw.net/api/channel/hls/randomchannel.m3u8?allow_source=true&player_backend=mediaplayer&sig=sig1&token=%7B%22adblock%22');
            video.dispatchEvent(new Event('encrypted'));
            await flushPromises();

            // TODO No idea why doing this via setupNavigatorToSupportKeySystem(KEY_SYSTEMS.WIDEVINE, true, false)
            keySession.update = jest.fn().mockReturnValue(Promise.reject());

            // The flushPromises above is important because the keysession object is created after setmediakeys
            // const updateSpy = jest.spyOn(keySession, 'update');
            keySession.dispatchEvent(new Event('message'), {
                message: new ArrayBuffer(5),
            });
            await flushPromises();

            expect(listener.onSinkError).toBeCalledWith(Object.assign({ code: 0 }, ERRORS.SESSION_UPDATE));
        });
    });
});
