import {
    AUTH_XML_URL,
    CDMError,
    ERRORS,
    KEY_SYSTEMS,
    KEY_SYSTEMS_BY_STRING,
    KeySystem,
} from './drm/constants';

import {
    arrayBuffersEqual,
    contentIdFromInitData,
    decodeBase64,
    encodeBase64,
    getParamsFromUrl,
    httpRequest,
    parsePSSHSupportFromInitData,
} from './drm/utils';

import { licenseRequestData } from './drm/PlayReady';

import { MediaSinkListener } from './mediasink';

/* TODO Robustness levels for Chrome best practices
    Spec notes that:
        robustness of type DOMString, defaulting to ""
        The robustness level associated with the content type.
        The empty string indicates that any ability to decrypt
        and decode the content type is acceptable.

    If we get requirements, we can set it to one of the settings below:
    https://storage.googleapis.com/wvdocs/Chrome_EME_Changes_and_Best_Practices.pdf
    Definition           EME Level     Widevine Device Security Level
    SW_SECURE_CRYPTO     1             3
    SW_SECURE_DECODE     2             3
    HW_SECURE_CRYPTO     3             2
    HW_SECURE_DECODE     4             1
    HW_SECURE_ALL        5             1
*/

// this is only used for W3C spec following EME feature check
const supportedConfig: Array<MediaKeySystemConfiguration> = [{
    initDataTypes: ['cenc'],
    audioCapabilities: [{
        contentType: 'audio/mp4;codecs="mp4a.40.2"',
    }],
    videoCapabilities: [{
        // comment out robustness for testing clearkey
        robustness: 'SW_SECURE_CRYPTO',
        contentType: 'video/mp4;codecs="avc1.42E01E"',
    }],
}];

export interface DRMManagerConfig {
    video: HTMLVideoElement;
    listener: MediaSinkListener;
}

/**
 * DRMManager sets up and handles media that contains DRM encryption
 * @param {Object} config
 * @param {HTMLElement} config.video - video element
 * @param {MediaSinkListener} config.listener - a collection of callback to repond playback changes
 */
export class DRMManager {
    private readonly video: HTMLVideoElement;
    private readonly listener: MediaSinkListener;
    private cdmSupport: Array<KeySystem> | null;
    private selectedCDM: KeySystem;
    private mediaKeys: MediaKeys;
    private pendingSessions: Array<MediaEncryptedEvent>;
    private sessions: Array<MediaEncryptedEvent>;
    private authXml: Promise<string>;
    constructor(config: DRMManagerConfig) {
        this.video = config.video;
        this.listener = config.listener;
        this.cdmSupport = null;
        this.selectedCDM = null;
        this.mediaKeys = undefined; // we will reserve null
        this.pendingSessions = [];
        this.reset();

        this.video.addEventListener('encrypted', (event) => this.handleEncrypted(event));
        this.video.addEventListener('webkitneedkey', (event: MediaEncryptedEvent) => this.handleSafariEncrypted(event));
    }

    configure(path: string) {
        // Only request authxml once when first configuring drm
        if (!this.authXml) {
            const parsed = new URL(path);
            const parts = parsed.pathname.split('/');
            const filename = parts[parts.length - 1];
            const channelName = filename.split('.')[0]; // remove extension

            const params = getParamsFromUrl(path);
            const token = params['token'];
            const sig = params['sig'];
            const authUrl = `${AUTH_XML_URL}${channelName}?token=${encodeURIComponent(token)}&sig=${sig}`;

            this.authXml = httpRequest(authUrl, {
                method: 'GET',
                responseType: 'text',
            }).catch((status) => {
                this.handleError(Object.assign({ code: status }, ERRORS.AUTH_XML_REQUEST));
            });
        }
    }

    reset() {
        this.authXml = null;
        this.sessions = [];
    }

    isProtected() {
        return this.authXml !== null;
    }

    /**
     * Ensure that uncaught errors are sent in the correct format.
     * Most issues should be a constant error found in ERRORS, but
     * in case we have an issue we weren't expecting, we should handle
     * the error in the same format. Block any errors if we're no
     * longer playing a DRM stream.
     */
    private handleError(err: CDMError) {
        this.listener.onSinkError({
            value: err.value || 4, /*NOT_SUPPORTED*/
            code: err.code || 0,
            message: err.message || '',
        });
    }

    /**
     * Checks to see if system is already handling
     * a session that matches initData
     * @param {ArrayBuffer} initData
     */
    private hasSession(initData: ArrayBuffer) {
        for (const session of this.sessions) {
            if (!session.initData) { continue; }
            if (arrayBuffersEqual(session.initData, initData)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Builds a promise catch chain to feature detect a keysystem that works.
     * This will get refactored once we have real systems working and we know
     * which format is requested/returned
     */
    private createKeySystemSupportChain(): Promise<MediaKeySystemAccess> {
        if (this.cdmSupport === null || this.cdmSupport.length === 0) {
            return Promise.reject(ERRORS.NO_PSSH_FOUND);
        }
        let promise: Promise<MediaKeySystemAccess> = Promise.reject();
        this.cdmSupport.forEach((cdm) => {
            promise = promise.catch(() => {
                return navigator.requestMediaKeySystemAccess(cdm.keySystem, supportedConfig);
            });
        });

        promise = promise.catch(() => {
            return Promise.reject(ERRORS.NO_CDM_SUPPORT);
        });

        return promise;
    }

    /**
     * Handles embeded DRM in initial video file
     * @param {MediaEncryptedEvent} event - event [https://www.w3.org/TR/encrypted-media/#dom-mediaencryptedevent]
     */
    private async handleEncrypted(event: MediaEncryptedEvent) {
        // if we already have this same session setup, ignore this event;
        if (this.hasSession(event.initData)) {
            return;
        }
        this.sessions.push(event);

        if (this.cdmSupport === null) {
            this.cdmSupport = parsePSSHSupportFromInitData(event.initData);
        }

        // if mediakeys have not started
        if (typeof this.mediaKeys === 'undefined') {
            // TODO there is a better way to check/manage state instead of using undefined -> null as loading
            // this will make sure things will not fire twice, since there is async that could be happening.
            this.mediaKeys = null;

            // create a promise chain of keySystem support
            try {
                const keySystemAccess: MediaKeySystemAccess = await this.createKeySystemSupportChain();
                this.selectedCDM = KEY_SYSTEMS_BY_STRING[keySystemAccess.keySystem];
                const keys: MediaKeys = await keySystemAccess.createMediaKeys();
                await this.setMediaKeys(keys);
            } catch (err) {
                this.handleError(err);
            }
        }

        this.addSession(event);
    }

    /**
     * Stores and sets mediaKeys/certificate
     * It will also create sessions for any sessions that are pending to be created
     * @param {Object} mediaKeys - MediaKeys [https://www.w3.org/TR/encrypted-media/#dom-mediakeys]
     */
    private setMediaKeys(mediaKeys: MediaKeys) {
        this.mediaKeys = mediaKeys;
        this.pendingSessions.forEach((session) => this.createSessionRequest(session));
        this.pendingSessions = [];
        return this.video.setMediaKeys(this.mediaKeys);
    }

    /**
     * Creates Sessions if MediaKeys is ready, otherwise it stores data
     * to create session once the MediaKeys is ready.
     * @param {MediaEncryptedEvent} event - event [https://www.w3.org/TR/encrypted-media/#dom-mediaencryptedevent]
     */
    private addSession(event: MediaEncryptedEvent) {
        if (this.mediaKeys) {
            this.createSessionRequest(event).
            catch (() => {
                this.handleError(ERRORS.KEY_SESSION_CREATION);
            });
        } else {
            this.pendingSessions.push(event);
        }
    }

    /**
     * Creates key session, prepares event handling of sessions messages,
     * and then generates a key session request.
     * {string} initDataType - [https://www.w3.org/TR/encrypted-media/#dom-mediaencryptedevent-initdatatype]
     * {ArrayBuffer} initData - [https://www.w3.org/TR/encrypted-media/#dom-mediaencryptedeventinit-initdata]
     * @param {MediaEncryptedEvent} event - event [https://www.w3.org/TR/encrypted-media/#dom-mediaencryptedevent]
     */
    private createSessionRequest({ initDataType, initData }: MediaEncryptedEvent) {
        const keySession: MediaKeySession = this.mediaKeys.createSession();
        keySession.addEventListener('message', (event: MediaKeyMessageEvent) => this.handleMessage(event));
        keySession.addEventListener('keystatuseschange', (event: Event) => this.handleKeyStatusesChange(event, initData));
        return keySession.generateRequest(initDataType, initData);
    }

    /**
     * Handles the event of a key changing, will be used for expiring and removing
     * key sessions.
     * @param {Object} event - Event
     * @param {ArrayBuffer} initData - ArrayBuffer [https://www.w3.org/TR/encrypted-media/#dom-mediaencryptedevent-initdata]
     */
    private handleKeyStatusesChange(event: Event, initData: ArrayBuffer) {
        const keySession = event.target as MediaKeySession;
        let expired = false;

        // based on https://www.w3.org/TR/encrypted-media/#example-using-all-events
        keySession.keyStatuses.forEach((status: MediaKeyStatus) => {
            switch (status) {
            case 'expired':
                // "All other keys in the session must have this status."
                // https://www.w3.org/TR/encrypted-media/#dom-mediakeystatus-expired
                expired = true;
                break;
            case 'internal-error':
                // https://www.w3.org/TR/encrypted-media/#dom-mediakeystatus-internal-error
                this.handleError(ERRORS.KEY_SESSION_INTERNAL);
                break;
            default:
                break;
            }
        });

        if (expired) {
            keySession.close().then(() => this.removeSession(initData));
        }
    }

    /**
     * Removes a session that matches initData
     * @param {ArrayBuffer} initData - [https://www.w3.org/TR/encrypted-media/#dom-mediaencryptedevent-initdata]
     */
    private removeSession(initData) {
        for (let i = 0; i < this.sessions.length; i++) {
            const session = this.sessions[i];
            if (session.initData === initData) {
                this.sessions.splice(i, 1);
                return;
            }
        }
    }

    /**
     * Handles key session 'message' event and generates/updates
     * license
     * @param {Event} event - [https://www.w3.org/TR/encrypted-media/#dom-mediakeymessageevent]
     */
    private async handleMessage(event: MediaKeyMessageEvent) {
        // grabs relevant session
        const keySession = event.target as MediaKeySession;

        const license = await this.generateLicense(event.message);
        keySession.update(license).
        catch(() => {
            this.handleError(ERRORS.SESSION_UPDATE);
        });
    }

    /**
     * Currently a ClearKey license generation
     * @param {Object} message - Message returned from CDM message event
     */
    private async generateLicense(message: ArrayBuffer) {
        if (this.selectedCDM === KEY_SYSTEMS.CLEARKEY) {
            // clearkey implementation where KID is key
            const request = JSON.parse(new TextDecoder().decode(message));

            const keys = request.kids.map((keyid) => {
                return { kty: 'oct', alg: 'A128KW', kid: keyid, k: keyid };
            });

            const result = new TextEncoder().encode(JSON.stringify({
                keys,
            }));
            return Promise.resolve(result);
        } else if (this.authXml) {
            const authXmlData = await this.authXml;
            return this.requestLicense(message, authXmlData);
        } else {
            this.handleError(ERRORS.AUTH_XML_REQUEST);
        }
    }

    private async requestLicense(message: ArrayBuffer, authXml: string) {
        const options = {
            method: 'POST',
            responseType: 'arraybuffer',
            body: message,
            headers: {
                'customdata': authXml,
                'Content-Type': 'application/octet-stream',
            },
        };

        // get additional data for specifics CDM license request calls
        if (this.selectedCDM === KEY_SYSTEMS.PLAYREADY) {
            const additionalData = licenseRequestData(message);
            options.body = additionalData.body;
            options.headers = Object.assign(options.headers, additionalData.headers);
        }

        return httpRequest(this.selectedCDM.licenseUrl, options).
        catch(() => {
            this.handleError(Object.assign({ code: status }, ERRORS.LICENSE_REQUEST));
        });
    }

    // SAFARI FAIRPLAY SUPPORT
    private async handleSafariEncrypted(event: MediaEncryptedEvent) {
        this.selectedCDM = KEY_SYSTEMS.FAIRPLAY;
        try {
            const certificate: ArrayBuffer = await httpRequest(KEY_SYSTEMS.FAIRPLAY.certUrl, {
                method: 'GET',
                responseType: 'arraybuffer',
                headers: {
                    'Pragma': 'Cache-Control: no-cache',
                    'Cache-Control': 'max-age=0',
                },
            });

            this.setupSafariMediaKeys(event, certificate).
            catch((err) => this.handleError(err));
        } catch (status) {
            this.handleError(Object.assign({ code: status }, ERRORS.CERT_REQUEST));
        }
    }

    /**
     * Safari's 'encrypted' initialization event. This works to
     * start initialization
     */
    private setupSafariMediaKeys(evt: MediaEncryptedEvent, certificate: ArrayBuffer) {
        return new Promise((resolve, reject) => {
            if (!this.video.webkitKeys) {
                this.video.webkitSetMediaKeys(new window.WebKitMediaKeys(KEY_SYSTEMS.FAIRPLAY.keySystem));
            }

            if (!this.video.webkitKeys) {
                reject('Issue setting fairplay media keys');
            }

            // Get the KeyID
            const contentId = contentIdFromInitData(evt.initData);

            const keySession = this.video.webkitKeys.createSession('video/mp4', evt.initData);

            if (!keySession) {
                return reject('Could not create key session');
            }

            keySession.contentId = contentId;

            keySession.addEventListener('webkitkeymessage', (event: WebKitMediaKeyMessageEvent) => {
                // tslint:disable-next-line no-any
                const session: any = event.target;
                const message = event.message;

                if ((String.fromCharCode.apply(null, event.message)) === 'certificate') {
                    session.update(new Uint8Array(certificate));
                } else {
                    // get license, and keySession.update()
                    this.getWebkitLicense(message, contentId).
                    then((license) => {
                        let keyText = license.trim();
                        if (keyText.substr(0, 5) === '<ckc>' && keyText.substr(-6) === '</ckc>') {
                            keyText = keyText.slice(5, -6);
                        }
                        session.update(decodeBase64(keyText));
                    }).
                    catch(reject);
                }
            });
            keySession.addEventListener('webkitkeyadded', resolve);
            keySession.addEventListener('webkitkeyerror', reject);
        });
    }

    /**
     * Get the webkit license
     * @param {Object} keyMessageEvent - Message event from current session
     */
    private getWebkitLicense(message, contentId) {
        if (!this.authXml) {
            return Promise.reject(ERRORS.AUTH_XML_REQUEST);
        }
        return this.authXml.then((authXml) => {
            const licenseUrl = KEY_SYSTEMS.FAIRPLAY.licenseUrl;
            const body = `spc=${encodeBase64(message)}&assetId=${contentId}`;
            const options = {
                method: 'POST',
                body,
                responseType: 'text',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                    'customdata': authXml,
                },
            };
            return httpRequest(licenseUrl, options).
            catch((status) => {
                return Promise.reject(Object.assign({ code: status }, ERRORS.LICENSE_REQUEST));
            });
        });
    }
}
