import { KEY_SYSTEMS, KeySystem } from './constants';

// some utils from https://github.com/videojs/videojs-contrib-eme/
// should probably revise these if our needs differ
export function arrayBuffersEqual(arrayBuffer1, arrayBuffer2): boolean {
    arrayBuffer1 = arrayBufferFrom(arrayBuffer1);
    arrayBuffer2 = arrayBufferFrom(arrayBuffer2);

    if (arrayBuffer1 === arrayBuffer2) {
        return true;
    }

    if (arrayBuffer1.byteLength !== arrayBuffer2.byteLength) {
        return false;
    }

    const dataView1 = new DataView(arrayBuffer1);
    const dataView2 = new DataView(arrayBuffer2);

    for (let i = 0; i < dataView1.byteLength; i++) {
        if (dataView1.getUint8(i) !== dataView2.getUint8(i)) {
            return false;
        }
    }
    return true;
}

function arrayBufferFrom(bufferOrTypedArray): ArrayBuffer {
    if (bufferOrTypedArray instanceof Uint8Array ||
        bufferOrTypedArray instanceof Uint16Array) {
        return bufferOrTypedArray.buffer;
    }
    return bufferOrTypedArray;
}

/**
 * Takes in array of PSSH data and returns KEY_SYSTEM objects
 * that are supported.
 *
 * @param {Array} psshArrayBuffer
 */
export function parsePSSHSupportFromInitData(initData): Array<KeySystem> {
    const uuids = parseAllPSSHData(initData);
    return mapPSSHSystemIds(uuids);
}

function mapPSSHSystemIds(uuids: Array<string>): Array<KeySystem> {
    const supportedKeySystems = [];
    uuids.forEach((uuid) => {
        Object.keys(KEY_SYSTEMS).forEach((key) => {
            const ks = KEY_SYSTEMS[key];
            if (ks.uuid === uuid) {
                supportedKeySystems.push(ks);
            }
        });
    });
    return supportedKeySystems;
}

/**
 * This is currently unused, but it allows you to pass in `encrypted`
 * event.initData and it will return PSSH data by key system uuid
 * This is from the Dash.js implementation to dynamically read
 * media's CDM support
 *
 * @param {ArrayBuffer} data - event.initData
 */
function parseAllPSSHData(data) {
    if (data === null) {
        return [];
    }

    const dv = new DataView(data.buffer || data); // data.buffer first for Uint8Array support
    const done = false;
    const uuids = [];

    // TODO: Need to check every data read for end of buffer
    let byteCursor = 0;
    while (!done) {
        let size;
        let nextBox;
        let version;
        let systemID;

        if (byteCursor >= dv.buffer.byteLength) {
            break;
        }

        /* Box size */
        size = dv.getUint32(byteCursor);
        nextBox = byteCursor + size;
        byteCursor += 4;

        /* Verify PSSH */
        if (dv.getUint32(byteCursor) !== boxTypeToInt('pssh')) {
            byteCursor = nextBox;
            continue;
        }
        byteCursor += 4;

        /* Version must be 0 or 1 */
        version = dv.getUint8(byteCursor);
        if (version !== 0 && version !== 1) {
            byteCursor = nextBox;
            continue;
        }
        byteCursor++;

        byteCursor += 3; /* skip flags */

        // 16-byte UUID/SystemID
        systemID = '';
        for (let i = 0; i < 4; i++) {
            systemID += byteToHex(dv.getUint8(byteCursor + i));
        }
        byteCursor += 4;
        systemID += '-';
        for (let i = 0; i < 2; i++) {
            systemID += byteToHex(dv.getUint8(byteCursor + i));
        }
        byteCursor += 2;
        systemID += '-';
        for (let i = 0; i < 2; i++) {
            systemID += byteToHex(dv.getUint8(byteCursor + i));
        }
        byteCursor += 2;
        systemID += '-';
        for (let i = 0; i < 2; i++) {
            systemID += byteToHex(dv.getUint8(byteCursor + i));
        }
        byteCursor += 2;
        systemID += '-';
        for (let i = 0; i < 6; i++) {
            systemID += byteToHex(dv.getUint8(byteCursor + i));
        }
        byteCursor += 6;

        systemID = systemID.toLowerCase();

        /* PSSH Data Size */
        const psshDataSize = dv.getUint32(byteCursor);
        byteCursor += 4;

        /* PSSH Data */
        uuids.push(systemID);
        byteCursor = nextBox;
    }

    return uuids;
}

// tslint:disable-next-line:no-any
export function httpRequest(url, options): Promise<any> {
    return new Promise((resolve, reject) => {
        const xhr = new XMLHttpRequest();
        xhr.open(options.method, url, true);

        for (const h in options.headers) {
            if (options.headers.hasOwnProperty(h)) {
                xhr.setRequestHeader(h, options.headers[h]);
            }
        }
        xhr.responseType = options.responseType;

        xhr.onload = () => {
            if (xhr.status === 200) {
                resolve(xhr.response);
            }
        };

        xhr.onloadend = () => {
            reject(xhr.status);
        };

        xhr.send(options.body);
    });
}

/**
 *
 * initData is an ascii encoded json object as follows
 * {
 *     "sinf":[...base64sinfbody...]
 * }
 */
export function contentIdFromInitData(initData) {
    const parsedInit = JSON.parse(String.fromCharCode.apply(null, initData));
    const sinf = decodeBase64(parsedInit.sinf[0]);
    const schi = searchForBox(sinf, 'schi');
    const tenc = searchForBox(schi, 'tenc');
    return arrayToHex(tenc.subarray(8, 24));
}

function searchForBox(buffer, boxType) {
    const view = new DataView(buffer.buffer, buffer.byteOffset, buffer.byteLength);
    const boxTypeValue = boxTypeToInt(boxType);
    let offset = 0;

    while (offset < buffer.byteLength) {
        const size = view.getUint32(offset);
        const type = view.getUint32(offset + 4);
        if (type === boxTypeValue) {
            return buffer.subarray(offset + 8, offset + size);
        }
        offset += size;
    }

    return new Uint8Array(buffer);
}

function boxTypeToInt(name) {
    return (
        (name.charCodeAt(0) << 24) +
        (name.charCodeAt(1) << 16) +
        (name.charCodeAt(2) << 8) +
        name.charCodeAt(3)
    );
}

function arrayToHex(buffer) {
    let hex = '';
    for (const buf of buffer) {
        hex += byteToHex(buf);
    }
    return hex;
}

function byteToHex(byte) {
    const hex = byte.toString(16);
    return (hex.length === 1 ? '0' + hex : hex);
}

export function decodeBase64(input) {
    const raw = atob(input);
    const rawLength = raw.length;
    const array = new Uint8Array(rawLength);
    for (let i = 0; i < rawLength; i++) {
        array[i] = raw.charCodeAt(i);
    }
    return array;
}

export function encodeBase64(input) {
    return btoa(String.fromCharCode.apply(null, input));
}

function decodeQSValue(value) {
  // decodeURIComponent does not convert + to space, so we have to do that ourselves
    return decodeURIComponent(value.replace(/\+/g, ' '));
}

export function getParamsFromUrl(url) {
    const urlObject: URL = new URL(url);
    const searchParams: URLSearchParams = urlObject.searchParams;
    const params = {};

    searchParams.forEach((value, key) => {
        params[decodeQSValue(key)] = value ? decodeQSValue(value) : '';
    });

    return params;
}
