import assert from 'assert';
import { Validator } from 'jsonschema';
import { PlayerState } from '../../event/state';
import { BUFFER_EMPTY_SCHEMA } from '../schemas/buffer-empty-schema';
import {
    checkPlayerState,
    emulateNetworkWithLatency,
    load,
    pause,
    play,
    seekTo,
    trackAnalyticEvent,
 } from '../utils';

const LATENCY_UP = {
    offline: false,
    latency: 150,
    uploadThroughput: -1,
    downloadThroughput: -1,
};
const LATENCY_DOWN = {
    offline: false,
    latency: 0,
    uploadThroughput: -1,
    downloadThroughput: -1,
};

const LIVE_URL = process.env.PLAYER_LIVEURL ? process.env.PLAYER_LIVEURL : 'https://twitch.tv/bufferbunny';
const VOD_URL = process.env.PLAYER_VODURL ? process.env.PLAYER_VODURL : 'https://www.twitch.tv/videos/306892197';
console.log(`Live URL: ${LIVE_URL}`);
console.log(` VOD URL: ${VOD_URL}`);

const BUFFER_EMPTY = 'buffer-empty';

describe('Buffer Empty event', () => {
    function testBufferEmpty(url: string) {
        beforeEach(() => {
            load(url);
            play();
            // allow time for playback to begin before throttling connection
            assert.ok(checkPlayerState(PlayerState.PLAYING));

            // @ts-ignore <https://gitter.im/webdriverio/webdriverio?at=5a830aa681cd9a5d7ea3f837>
            browser.cdp('Network', 'enable');
        });

        afterEach(() => {
            // @ts-ignore <https://gitter.im/webdriverio/webdriverio?at=5a830aa681cd9a5d7ea3f837>
            browser.cdp('Network', 'disable');
        });

        it(`should fire when video begins buffering with base properties on ${url}`, () => {
            // Inducing buffer
            emulateNetworkWithLatency(LATENCY_UP);

            const bufferEmpty = trackAnalyticEvent(BUFFER_EMPTY);
            assert.equal(bufferEmpty['name'], BUFFER_EMPTY);

            // Restoring connectivity
            emulateNetworkWithLatency(LATENCY_DOWN);

            const schemaComparison = new Validator().validate(bufferEmpty, BUFFER_EMPTY_SCHEMA);
            assert.equal(schemaComparison.errors.length, 0, `Schema Error: ${schemaComparison.errors}`);
        });

        it(`should not fire again if video is paused then unpaused while still buffering for ${url}`, () => {
            // Inducing buffer
            emulateNetworkWithLatency(LATENCY_UP);

            const bufferEmpty = trackAnalyticEvent(BUFFER_EMPTY);

            pause();
            assert.ok(checkPlayerState(PlayerState.IDLE));
            play();

            // Should not have a new buffer empty fired
            assert.throws(() => { trackAnalyticEvent(BUFFER_EMPTY, 3000); });

            assert.equal(bufferEmpty['name'], BUFFER_EMPTY);

            // Restoring connectivity
            emulateNetworkWithLatency(LATENCY_DOWN);
        });

        it(`buffer_empty_count property is accurate in event payload on consecutive event firings for ${url}`, () => {
            const bufferEmpty = [];

            // @ts-ignore <https://gitter.im/webdriverio/webdriverio?at=5a830aa681cd9a5d7ea3f837>
            for (let iteration = 0; iteration < 2; iteration++) {
                // Inducing buffer
                emulateNetworkWithLatency(LATENCY_UP);

                bufferEmpty.push(trackAnalyticEvent(BUFFER_EMPTY));

                // Restoring connectivity
                emulateNetworkWithLatency(LATENCY_DOWN);

                // allow time for playback to begin before throttling connection in the next iteration
                assert.ok(checkPlayerState(PlayerState.PLAYING));
                assert.equal(bufferEmpty[iteration].properties.buffer_empty_count, iteration + 1);
            }
        });
    }
    testBufferEmpty(LIVE_URL);
    testBufferEmpty(VOD_URL);

    it(`should not fire when seeking for ${VOD_URL}`, () => {
        load(VOD_URL);
        play();
        assert.ok(checkPlayerState(PlayerState.PLAYING));
        seekTo(80);
        assert.throws(() => { trackAnalyticEvent(BUFFER_EMPTY, 4000); });
    });
});
