import assert from 'assert';
import { Validator } from 'jsonschema';
import { PlayerState } from '../../event/state';
import { BUFFER_REFILL_SCHEMA } from '../schemas/buffer-refill-schema';
import {
    between,
    checkPlayerState,
    emulateNetworkWithLatency,
    load,
    play,
    seekTo,
    trackAnalyticEvent,
} from '../utils';

const LATENCY_UP = {
    offline: false,
    latency: 150,
    uploadThroughput: -1,
    downloadThroughput: -1,
};
const LATENCY_DOWN = {
    offline: false,
    latency: 0,
    uploadThroughput: -1,
    downloadThroughput: -1,
};

const LIVE_URL = process.env.PLAYER_LIVEURL ? process.env.PLAYER_LIVEURL : 'https://twitch.tv/bufferbunny';
const VOD_URL = process.env.PLAYER_VODURL ? process.env.PLAYER_VODURL : 'https://www.twitch.tv/videos/306892197';
console.log(`Live URL: ${LIVE_URL}`);
console.log(` VOD URL: ${VOD_URL}`);

const BUFFER_REFILL = 'buffer-refill';

// TODO: VP-6497
describe.skip('Buffer Refill event', () => {
    function testBufferRefill(url: string) {
        beforeEach(() => {
            load(url);
            play();
            // allow time for playback to begin before throttling connection
            assert.ok(checkPlayerState(PlayerState.PLAYING));
            // @ts-ignore <https://gitter.im/webdriverio/webdriverio?at=5a830aa681cd9a5d7ea3f837>
            browser.cdp('Network', 'enable');
        });

        afterEach(() => {
            // @ts-ignore <https://gitter.im/webdriverio/webdriverio?at=5a830aa681cd9a5d7ea3f837>
            browser.cdp('Network', 'disable');
        });

        it(`should fire when video begins buffering with base properties on ${url}`, () => {
            // Inducing buffer
            emulateNetworkWithLatency(LATENCY_UP);
            assert.ok(checkPlayerState(PlayerState.BUFFERING));

            emulateNetworkWithLatency(LATENCY_DOWN);

            const bufferRefill = trackAnalyticEvent(BUFFER_REFILL);

            const schemaComparison = new Validator().validate(bufferRefill, BUFFER_REFILL_SCHEMA);
            assert.equal(schemaComparison.errors.length, 0, `Schema Error: ${schemaComparison.errors}`);
        });

        it(`buffering_time property is accurate in event payload for ${url}`, () => {
            // Inducing buffer
            emulateNetworkWithLatency(LATENCY_UP);
            assert.ok(checkPlayerState(PlayerState.BUFFERING));
            const bufferingEventTimeStamp = Date.now() / 1000;
            emulateNetworkWithLatency(LATENCY_DOWN);

            const bufferRefill = trackAnalyticEvent(BUFFER_REFILL);
            checkPlayerState(PlayerState.PLAYING);
            const playingEventTimeStamp = Date.now() / 1000;
            const difference = bufferRefill['properties']['buffering_time'] - (playingEventTimeStamp - bufferingEventTimeStamp);
            assert.ok(between(difference, -1, 1), `Expected between - 1 & 1, got: ${difference}`);
        });

        it(`buffer_empty_count property should be accurate in event payload on consecutive event firings for ${url}`, () => {
            for (let iteration = 0; iteration < 2; iteration++) {
                // Inducing buffer
                emulateNetworkWithLatency(LATENCY_UP);
                assert.ok(checkPlayerState(PlayerState.BUFFERING));

                emulateNetworkWithLatency(LATENCY_DOWN);
                const bufferRefillEvent = trackAnalyticEvent(BUFFER_REFILL);

                // allow time for playback to begin before throttling connection in the next iteration
                assert.ok(checkPlayerState(PlayerState.PLAYING));
                assert.equal(bufferRefillEvent['properties']['buffer_empty_count'], iteration + 1);
            }
        });
    }

    testBufferRefill(VOD_URL);
    testBufferRefill(LIVE_URL);

    it(`should not fire when seeking for ${VOD_URL}`, () => {
        load(VOD_URL);
        play();
        assert.ok(checkPlayerState(PlayerState.PLAYING));
        seekTo(80);
        assert.throws(() => { trackAnalyticEvent(BUFFER_REFILL, 4000); });
    });
});
