import assert from 'assert';
import { Validator } from 'jsonschema';
import { PlayerState } from '../../event/state';
import { VIDEO_PLAY_SCHEMA } from '../schemas/video-play-schema';
import {
    checkPlayerState,
    load,
    pause,
    play,
    setAutoplay,
    trackAnalyticEvent,
} from '../utils';

const LIVE_URL = process.env.PLAYER_LIVEURL ? process.env.PLAYER_LIVEURL : 'https://twitch.tv/bufferbunny';
const VOD_URL = process.env.PLAYER_VODURL ? process.env.PLAYER_VODURL : 'https://www.twitch.tv/videos/306892197';
console.log(`Live URL: ${LIVE_URL}`);
console.log(` VOD URL: ${VOD_URL}`);

const VIDEO_PLAY = 'video-play';

describe('Video-Play', () => {
    function testVideoPlay(url: string) {
        it(`should fire first time play with base properties on loading ${url} autoplaying player`, () => {
            setAutoplay(true);
            load(url);
            const videoPlay = trackAnalyticEvent(VIDEO_PLAY);
            assert.equal(videoPlay['name'], VIDEO_PLAY);

            const schemaComparison = new Validator().validate(videoPlay, VIDEO_PLAY_SCHEMA);
            assert.equal(schemaComparison.errors.length, 0, `Schema Error: ${schemaComparison.errors}`);
        });

        it(`should fire first time play with base properties on loading ${url} on non autoplaying player`, () => {
            load(url);
            play();
            const videoPlay = trackAnalyticEvent(VIDEO_PLAY);
            assert.equal(videoPlay['name'], VIDEO_PLAY);

            const schemaComparison = new Validator().validate(videoPlay, VIDEO_PLAY_SCHEMA);
            assert.equal(schemaComparison.errors.length, 0, `Schema Error: ${schemaComparison.errors}`);
        });

        it(`should not fire after loading ${url} on a non autoplaying player`, () => {
            load(url);
            trackAnalyticEvent(VIDEO_PLAY);
            assert.throws(() => { trackAnalyticEvent(VIDEO_PLAY, 10000); });
        });

        it(`should not be fired after un-pausing video on ${url}`, () => {
            load(url);
            play();
            trackAnalyticEvent(VIDEO_PLAY);
            pause();
            assert.ok(checkPlayerState(PlayerState.IDLE));
            play();
            assert.throws(() => { trackAnalyticEvent(VIDEO_PLAY, 3000); });
        });

        it(`should be fired on first play with base properties on subsequent non-autoplaying loads on ${url}`, () => {
            load(url);
            play();
            trackAnalyticEvent(VIDEO_PLAY);
            load(url);
            play();
            const videoPlay = trackAnalyticEvent(VIDEO_PLAY);
            assert.equal(videoPlay['name'], VIDEO_PLAY);

            const schemaComparison = new Validator().validate(videoPlay, VIDEO_PLAY_SCHEMA);
            assert.equal(schemaComparison.errors.length, 0, `Schema Error: ${schemaComparison.errors}`);
        });

        it(`should be fired on first play with base properties on subsequent autoplaying loads on ${url}`, () => {
            setAutoplay(true);
            load(url);
            trackAnalyticEvent(VIDEO_PLAY);
            load(url);
            const videoPlay = trackAnalyticEvent(VIDEO_PLAY);
            assert.equal(videoPlay['name'], VIDEO_PLAY);

            const schemaComparison = new Validator().validate(videoPlay, VIDEO_PLAY_SCHEMA);
            assert.equal(schemaComparison.errors.length, 0, `Schema Error: ${schemaComparison.errors}`);
        });
    }
    testVideoPlay(LIVE_URL);
    testVideoPlay(VOD_URL);
});
