/**
 * Public events sent from the MediaPlayer. These are listened to internally,
 * but can also be publicly consumed with `addEventListener` All events are
 * currently emitted from the WebMediaPlayer in the worker. The getters that
 * are updated with each event are listed below. The payloads are currently not
 * publicly exposed, but contain the internal state change information.
 */
export enum PlayerEvent {
    /**
     * Emitted when the MediaPlayer instances has been created
     * Updated functions:
     *     getVersion()
     */
    INITIALIZED = 'PlayerInitialized',
    /**
     * Rendered quality has changed
     * Updated functions:
     *     getQuality()
     */
    QUALITY_CHANGED = 'PlayerQualityChanged',
    /**
     * Auto turned on or off
     * Updated functions:
     *     getAutoSwitchQuality()
     */
    AUTO_SWITCH_QUALITY_CHANGED = 'PlayerAutoSwitchQualityChanged',
    /**
     * Duration value has changed
     * Updated functions:
     *     getDuration()
     *     getStartOffset()
     */
    DURATION_CHANGED = 'PlayerDurationChanged',
    /**
     * Volume changed in range 0-1
     */
    VOLUME_CHANGED = 'PlayerVolumeChanged',
    /**
     * Muted state toggled
     */
    MUTED_CHANGED = 'PlayerMutedChanged',
    /**
     * The playback rate has changed
     */
    PLAYBACK_RATE_CHANGED = 'PlayerPlaybackRateChanged',
    /**
     * An errant BUFFERING occurred
     */
    REBUFFERING = 'PlayerRebuffering',
    /**
     * The browser blocked non-muted playback without a user gesture.
     * Mute and play to start playback or wait for user gesutre.
     */
    AUDIO_BLOCKED = 'PlayerAudioBlocked',
    /**
     * The browser blocked playback without a user gesture.
     * Mute and play to start playback or wait for user gesutre.
     */
    PLAYBACK_BLOCKED = 'PlayerPlaybackBlocked',
    /**
     * A fatal error occurred
     * @param {string} error.type - Error type from 'error.js'
     * @param {string} error.message - human readable description
     */
    ERROR = 'PlayerError',
    /**
     * A recoverable error occurred
     * @param {string} error.type - Error type from 'error.js'
     * @param {string} error.message - human readable description
     */
    RECOVERABLE_ERROR = 'PlayerRecoverableError',
    /**
     * A tracking event as been emited. Should be sent to Spade
     * @param {string} event.name - Name of tracking event
     * @param {object} event.properties - key-value pairs to store
     */
    TRACKING = 'PlayerTracking',
    /**
     * The playback timeupdate event - web only
     * Updated functions:
     *     getPosition()
     */
    TIME_UPDATE = 'PlayerTimeUpdate',
    /**
     * More data has been added to the end of the buffer
     * Updated value:
     *     getPosition() + getBufferDuration()
     *     getBuffered()
     */
    BUFFER_UPDATE = 'PlayerBufferUpdate',
    /**
     * A Seek request was completed
     * @param {number} time The position seek completed ended
     */
    SEEK_COMPLETED = 'PlayerSeekCompleted',
    /**
     * Profiler event with profiler data
     * @param {string} event The profiler event name
     */
    PROFILE = 'PlayerProfile',
    /**
     * Updates the twitch playlist tracking properties
     * @param {Object} playlist properties that have changed
     */
    SESSION_DATA = 'PlayerSessionData',
    /**
     * Notifies a failed API call inside the worker message handler
     * @param {string} message The message of exception
     */
    WORKER_ERROR = 'PlayerWorkerError',

    /**
     * Arbitrary timed metadata defined by the application
     * @param {string} metadata.type - the encoding of 'data'
     * @param {ArrayBuffer} - metadata.data - the raw binary
     */
    METADATA = 'PlayerMetadata',
    /**
     * Generic Timed metadata enter
     * @deprecated
     */
    METADATA_CUE_ENTER = 'PlayerMetadataCueEnter',
    /**
     * Generic Timed metadata exit
     * @deprecated
     */
    METADATA_CUE_EXIT = 'PlayerMetadataCueExit',
}
