/* tslint:disable:no-require-imports */
describe('fetchShim', () => {
    test('uses native fetch if it and ReadableStream are defined', () => {
        const fetchMock = jest.fn();
        global['fetch'] = fetchMock;
        global['ReadableStream'] = jest.fn();

        // This is needed to isolate the fetchShim instantiation
        // between this test and other tests
        let fetchShim;
        jest.isolateModules(() => {
            fetchShim = require('./fetch').fetchShim;
        });

        fetchShim();
        expect(fetchMock).toHaveBeenCalled();
    });

    test('uses fetchPolyfill if native fetch is not defined', () => {
        global['fetch'] = undefined;
        global['ReadableStream'] = jest.fn();

        let fetchImports;
        jest.isolateModules(() => {
            fetchImports = require('./fetch');
        });
        const fetchShim = fetchImports.fetchShim;
        const testImports = fetchImports._testExports;

        expect(fetchShim).toEqual(testImports.fetchPolyfill);
    });

    test('uses fetchPolyfill if native ReadableStream is not defined', () => {
        const fetchMock = jest.fn();
        global['fetch'] = fetchMock;
        global['ReadableStream'] = undefined;

        let fetchImports;
        jest.isolateModules(() => {
            fetchImports = require('./fetch');
        });
        const fetchShim = fetchImports.fetchShim;
        const testImports = fetchImports._testExports;

        expect(fetchShim).toEqual(testImports.fetchPolyfill);
    });
});

describe('fetchPolyfill', () => {
    let testImports;
    jest.isolateModules(() => {
        testImports = require('./fetch')._testExports;
    });

    let xhrAbortMock;
    let xhrAddEventListenerMock;
    let xhrOpenMock;
    let xhrRemoveEventListenerMock;
    let xhrSendMock;
    let xhrSetRequestHeaderMock;

    let xhrMock;

    beforeEach(() => {
        xhrAbortMock = jest.fn();
        xhrAddEventListenerMock = jest.fn();
        xhrOpenMock = jest.fn();
        xhrRemoveEventListenerMock = jest.fn();
        xhrSendMock = jest.fn();
        xhrSetRequestHeaderMock = jest.fn();

        xhrMock = {
            abort: xhrAbortMock,
            addEventListener: xhrAddEventListenerMock,
            open: xhrOpenMock,
            readyState: 0,
            removeEventListener: xhrRemoveEventListenerMock,
            send: xhrSendMock,
            setRequestHeader: xhrSetRequestHeaderMock,
            withCredentials: false,
        };
        global['XMLHttpRequest'] = jest.fn().mockImplementation(() => xhrMock);
    });

    test('uses given method to make request', () => {
        testImports.fetchPolyfill('twitch.tv', { method: 'POST' });
        expect(xhrOpenMock).toBeCalledWith('POST', 'twitch.tv');
    });

    test('uses GET as default method if none is provided', () => {
        testImports.fetchPolyfill('twitch.tv', {});
        expect(xhrOpenMock).toBeCalledWith('GET', 'twitch.tv');
    });

    test('adds headers when given', () => {
        const headers = {
            firstheader: 'first',
            secondheader: 'second',
        };
        testImports.fetchPolyfill('twitch.tv', { headers });
        expect(xhrSetRequestHeaderMock).toBeCalledWith('firstheader', 'first');
        expect(xhrSetRequestHeaderMock).toBeCalledWith('secondheader', 'second');
    });

    test('sets withCredentials property if set via options', () => {
        testImports.fetchPolyfill('twitch.tv', { credentials: 'include' });
        expect(xhrMock.withCredentials).toBe(true);
    });

    test('does not set withCredentials property if not set via options', () => {
        testImports.fetchPolyfill('twitch.tv', { credentials: 'omit' });
        expect(xhrMock.withCredentials).toBe(false);
    });

    test('request rejects when an error occurs', () => {
        const req = testImports.fetchPolyfill('twitch.tv', {});
        const errorCallback = xhrAddEventListenerMock.mock.calls.find((e) => {
            return e[0] === 'error';
        })[1];
        errorCallback();
        expect(req).rejects.toThrow('network error');
    });

    test('aborts request if abort signal is sent', () => {
        const abortController = new AbortController();
        const signal = abortController.signal;
        const req = testImports.fetchPolyfill('twitch.tv', { signal });
        abortController.abort();

        expect(req).rejects.toThrow('request aborted');
    });

    test('when readyState is not `HEADERS_RECEIVED`, does not resolve', () => {
        const req = testImports.fetchPolyfill('twitch.tv', {});
        const readyStateListener = xhrAddEventListenerMock.mock.calls.find((e) => {
            return e[0] === 'readystatechange';
        });
        const readyStateCallback = readyStateListener[1];
        xhrMock.readyState = 0;
        readyStateCallback();
        expect(req).resolves.not.toBeInstanceOf(testImports.ResponseShim);

        xhrMock.readyState = 1;
        readyStateCallback();
        expect(req).resolves.not.toBeInstanceOf(testImports.ResponseShim);

        xhrMock.readyState = 3;
        readyStateCallback();
        expect(req).resolves.not.toBeInstanceOf(testImports.ResponseShim);

        xhrMock.readyState = 4;
        readyStateCallback();
        expect(req).resolves.not.toBeInstanceOf(testImports.ResponseShim);
    });

    test('when readyState is `HEADERS_RECEIVED`, resolves with response', () => {
        const req = testImports.fetchPolyfill('twitch.tv', {});
        const readyStateListener = xhrAddEventListenerMock.mock.calls.find((e) => {
            return e[0] === 'readystatechange';
        });
        const readyStateCallback = readyStateListener[1];
        xhrMock.readyState = 2;
        readyStateCallback();

        expect(req).resolves.toBeInstanceOf(testImports.ResponseShim);
    });
});
