import Promise from './promise';
import { ReadableStreamShim, StreamingResponseType } from './readable-stream-shim';

// Export fetch or fetch-shim based on compatibility
export const fetchShim: Function = (self.fetch && self.ReadableStream) ? self.fetch.bind(self) : fetchPolyfill;

const responseType = ((self.fetch && self.ReadableStream) ? 'arraybuffer' :
    checkSupport('moz-chunked-arraybuffer')
    || checkSupport('ms-stream')
    || 'arraybuffer') as StreamingResponseType;

// Emulates 'fetch' with XMLHttpRequest
function fetchPolyfill(url: string, init: RequestInit = {}): Promise<ResponseShim> {

    return new Promise((resolve, reject) => {
        const xhr = new XMLHttpRequest();
        xhr.open(init.method || 'GET', url);

        for (const h in init.headers) {
            if (init.headers.hasOwnProperty(h)) {
                xhr.setRequestHeader(h, init.headers[h]);
            }
        }

        if (init.credentials === 'include') {
            xhr.withCredentials = true;
        }

        // Initialize stream for the response body
        const respBody = new ReadableStreamShim(xhr, responseType);

        const onResponse = () => {
            // readyState == HEADERS_RECIEVED
            if (xhr.readyState === 2) {
                xhr.removeEventListener('readystatechange', onResponse);
                resolve(new ResponseShim(xhr, respBody));
            }
        };

        // Resolve the promise when the respose has arrived
        xhr.addEventListener('readystatechange', onResponse);

        // Abort when signalled
        if (init.signal) {
            init.signal.onabort = () => {
                xhr.abort();
                const err = new Error('request aborted');
                err.name = 'AbortError';
                respBody.getReader().error(err);
                reject(err);
            };
        }

        // Shut down stream if request errors
        xhr.addEventListener('error', () => {
            const err = new Error('network error');
            respBody.getReader().error(err);
            reject(err);
        });

        xhr.send(init.body || null);
    });
}

class ResponseShim {
    body: ReadableStreamShim;
    status: number;
    headers: HeadersShim;

    constructor(xhr: XMLHttpRequest, stream: ReadableStreamShim) {
        // Emulate properties of the fetch 'Response' type.
        this.body = stream;
        this.status = xhr.status;
        this.headers = new HeadersShim(xhr);
    }
}

class HeadersShim {
    private xhr: XMLHttpRequest;

    constructor(xhr: XMLHttpRequest) {
        this.xhr = xhr;
    }

    get(key: string): string | null {
        return this.xhr.getResponseHeader(key);
    }
}

// Check if 'type' can be used as an XMLHttpRequest responseType
// If so, return it
function checkSupport(type: StreamingResponseType): string {
    try {
        const xhr = new XMLHttpRequest();
        xhr.open('GET', 'https://twitch.tv');
        xhr.responseType = type as XMLHttpRequest['responseType'];
        return xhr.responseType === type ? type : '';
    } catch (e) {
        return '';
    }
}

export const _testExports = {
    fetchPolyfill,
    checkSupport,
    ResponseShim,
};
