import 'whatwg-fetch';

jest.mock('./fetch', () => {
    return {
        fetchShim: jest.fn().mockResolvedValue({}),
    };
});

import { fetchShim } from './fetch';
import { _testExports, sendFetchRequest } from './http';
import { EmscriptenModule } from './web-mediaplayer';

const abortSignal = 'abortsignal';
const abortMock = jest.fn();
const abortControllerMock = jest.fn().mockImplementation(() => ({
    signal: abortSignal,
    abort: abortMock,
}));
global['AbortController'] = abortControllerMock;

describe('sendFetchRequest', () => {
    const requestProxy = {
        response: jest.fn(),
        error: jest.fn(),
        delete: jest.fn(),
    };
    const url = 'fakeurl';
    const timeout = 100;

    test('returns a function that will cancel the request', () => {
        const cancelFn = sendFetchRequest(undefined, requestProxy, url, {}, timeout);
        expect(typeof cancelFn).toBe('function');

        cancelFn();
        expect(abortMock).toBeCalled();
    });

    test('calls fetchshim with the url and request init object', () => {
        const initRequest: RequestInit = {
            headers: {},
        };

        sendFetchRequest(undefined, requestProxy, url, initRequest, timeout);
        expect(fetchShim).toBeCalledWith(url, initRequest);
    });
});

describe('ResponseController', () => {
    let abortController;
    let responseController;

    beforeEach(() => {
        abortController = new AbortController();
        responseController = new _testExports.ResponseController(undefined, abortController);

        abortMock.mockReset();
    });

    test('can construct ResponseController', () => {
        expect(responseController).toBeInstanceOf(_testExports.ResponseController);
    });

    test('abortController is set to passed in abort controller', () => {
        expect(responseController['abortController'].signal).toEqual(abortController.signal);
    });

    test('setResponse sets the response', () => {
        const response: Response = new Response();
        responseController.setResponse(response);
        expect(responseController['response']).toEqual(response);
    });

    test('setResponse sets pendingAbort to false if true and cancels the reader', () => {
        const response: Response = new Response();
        const mockReader = {
            cancel: jest.fn(),
        };

        responseController['pendingAbort'] = true;
        responseController['reader'] = mockReader;
        responseController.setResponse(response);

        expect(responseController['pendingAbort']).toEqual(false);
        expect(mockReader.cancel).toBeCalled();
    });

    test('abort calls abort on the provided abortController', () => {
        responseController.abort();
        expect(abortMock).toBeCalled();
    });

    test('abort calls cancel on the reader if the response already returned', () => {
        const cancelMock = jest.fn();
        const response = {
            body: {
                getReader: jest.fn().mockReturnValue({
                    cancel: cancelMock,
                }),
            },
        };

        responseController['response'] = response;
        responseController.abort();

        expect(cancelMock).toBeCalled();
    });

    test('cancel sets cancelled to true and calls abort', () => {
        responseController['abort'] = jest.fn();
        responseController.cancel();
        expect(responseController['cancelled']).toBe(true);
        expect(responseController['abort']).toBeCalled();
    });

    test('readBody calls read on the reader', () => {
        const mockRead = jest.fn().mockResolvedValue({
            done: true,
            value: {},
        });
        const mockReaderProxy = {
            end: jest.fn(),
        };

        responseController['reader'] = {
            read: mockRead,
        };
        responseController.readBody(mockReaderProxy);
        expect(mockRead).toBeCalled();
    });
});
