// tslint:disable-next-line: no-require-imports no-var-requires
import { fetchShim } from './fetch';
import Promise from './promise';
import { HttpResponse, ReaderProxy, RequestProxy } from './web-http';
import { EmscriptenModule } from './web-mediaplayer';

/**
 * Send a fetch request and proxy result back to emscripten
 * @param  {RequestProxy} requestProxy - defined in 'web-http.cpp'
 * @param  {string} url - url to request
 * @param  {object} opts - options for 'fetch'
 * @param  {number} timeout - milliseconds to wait for timeout
 */
export function sendFetchRequest(emModule: EmscriptenModule, requestProxy: RequestProxy, url: string, init: RequestInit, timeout: number) {
    let abortController = null;
    if (typeof AbortController !== 'undefined') {
        abortController = new AbortController();
        init.signal = abortController.signal;
    }

    const respCtrl = new ResponseController(emModule, abortController);
    const timeoutID = setTimeout(() => respCtrl.abort(), timeout);

    function cleanup() {
        clearTimeout(timeoutID);
        requestProxy.delete();
    }

    fetchShim(url, init).then((resp) => {
        respCtrl.setResponse(resp);
        if (!respCtrl.cancelled) {
            requestProxy.response(respCtrl);
        }
    }).catch((err) => {
        if (!respCtrl.cancelled) {
            // tslint:disable-next-line: no-console
            console.error('HTTP Response Error:', err);
            requestProxy.error(err.name === 'AbortError');
        }
    }).then(cleanup, cleanup);

    return () => respCtrl.cancel();
}

class ResponseController implements HttpResponse {
    cancelled: boolean;

    private readonly module: EmscriptenModule;
    private pendingAbort: boolean;
    private response: Response;
    private reader: ReadableStreamDefaultReader;
    private readonly abortController: AbortController | null;

    constructor(mod: EmscriptenModule, abortController: AbortController) {
        this.cancelled = false;
        this.module = mod;
        this.pendingAbort = false;
        this.response = null;
        this.reader = null;
        this.abortController = abortController;

        // // Emscripten isn't able to call prototype function with the
        // // correct value of 'this', so we'll prebind instead.
        this.readBody = this.readBody.bind(this);
    }

    setResponse(resp: Response) {
        this.response = resp;
        if (this.pendingAbort) {
            this.pendingAbort = false;
            this.getReader().cancel();
        }
    }

    abort() {
        if (this.response) {
            // Cancelling the reader if available guarantees
            // no https chunks will be received after this is called
            this.getReader().cancel();
        } else if (this.abortController) {
            this.abortController.abort();
        } else {
            // We'll abort when we receive the reponse
            this.pendingAbort = true;
        }
    }

    cancel() {
        this.cancelled = true;
        this.abort();
    }

    getHeader(key: string): string {
        try {
            return this.response.headers.get(key) || '';
        } catch (e) {
            return '';
        }
    }

    getStatus(): number {
        try {
            return this.response.status;
        } catch (e) {
            return 0;
        }
    }

    /**
     * Read the body of a 'Response' as a stream.
     * @param  {ReaderProxy} readerProxy - defined in 'web-http.cpp'
     * @param  {number} timeout - Abort reading if 'timeout' ms elapse with no data
     */
    readBody(readerProxy: ReaderProxy, timeout: number) {
        // Since scheduling and clearing a timeout every chunk
        // is a little expensive, poll at the 'timeout' interval
        // intead, rescheduling with the soonest possible timeout.
        let lastRead = performance.now();
        const probeTimeout = () => {
            const elapsed = performance.now() - lastRead;
            if (elapsed < timeout) {
                timeoutID = setTimeout(probeTimeout, timeout - elapsed);
            } else {
                // Need to call 'error' handler explicitly
                // since we're canceling the reader
                this.abort();
                readerProxy.error(true);
            }
        };
        let timeoutID = setTimeout(probeTimeout, timeout);

        const handleChunk = ({ done, value }: ReadableStreamReadResult<Uint8Array>) => {
            // Ignore any chunks after we've been cancelled
            if (this.cancelled) {
                return;
            }

            if (done) {
                readerProxy.end();
                return;
            }

            // Might get zero-length chunks if using fetch shim
            const byteLength = value.byteLength;
            if (byteLength) {
                readerProxy.read(copyToHeap(this.module, value), byteLength);
            }

            lastRead = performance.now();
            return this.getReader().read().then(handleChunk);
        };

        this.getReader().read().then(handleChunk).catch((err) => {
            // Cancelling doesn't cause this promise
            // to be rejected, so no need to check.
            // tslint:disable-next-line: no-console
            console.error('HTTP Read Error:', err);
            readerProxy.error(false);
        })
        .then(() => {
            clearTimeout(timeoutID);
            readerProxy.delete();
        });
    }

    private getReader(): ReadableStreamDefaultReader {
        if (!this.reader) {
            try {
                this.reader = this.response.body.getReader();
            } catch (e) {
                this.reader = new NullReader();
            }
        }
        return this.reader;
    }
}

class NullReader implements ReadableStreamDefaultReader {
    closed: Promise<void>;

    constructor() {
        this.closed = Promise.resolve();
    }

    read() {
        return Promise.resolve({ done: true } as ReadableStreamReadResult<Uint8Array>);
    }

    cancel() {
        return Promise.resolve();
    }

    releaseLock() {
        // noop
    }
}

/**
 * Copies a TypedArray to the emscripten heap.
 * The same buffer is used on every call, so the returned pointer
 * is valid until the next call of this function. The internal buffer
 * will be the size of the largest buffer passed to this function
 * @param  {TypedArray} buf - buffer to copy to heap
 * @return {Number} pointer into the emscripten heap of copied buffer
 */
const copyToHeap = (() => {
    let ptr = 0;
    let len = 0;
    // Emscripten definitions:
    // _malloc(n): allocates 'n' byte buffer in emscripten heap
    // _free(ptr): frees a buffer allocated by malloc
    // HEAPU8: A Uint8Array view into the emscripten heap
    return ({ HEAPU8, _free, _malloc }: EmscriptenModule, buf) => {
        const bufLen = buf.byteLength;
        if (bufLen > len) {

            if (ptr) {
                _free(ptr);
            }

            ptr = _malloc(bufLen);
            len = bufLen;
        }
        HEAPU8.set(buf, ptr);
        return ptr;
    };
})();

export const _testExports = {
    ResponseController,
};
